#include <mail/nwsmtp/src/web/request.h>
#include <mail/nwsmtp/src/utils.h>

#include <gtest/gtest.h>

using NNwSmtp::NWeb::ReadMultiPartBody;
using NNwSmtp::NWeb::ParseBody;
using NNwSmtp::NUtil::MakeSegment;

using ::testing::TestWithParam;

using TPartInfo = std::tuple<std::string, std::string, std::string>;
using TPartInfos = std::vector<TPartInfo>;


auto MakePart(const std::string& type, const std::string& subtype, const std::string& body) {
    ymod_webserver::content_part part;
    part.content.type = type;
    part.content.subtype = subtype;
    part.body = MakeSegment(body);
    return part;
}

auto MakeMultiPartBody(std::vector<TPartInfo> partInfos) {
    std::vector<ymod_webserver::content_part> parts;
    for (auto partInfo : partInfos) {
        parts.push_back(std::apply(MakePart, partInfo));
    }
    return parts;
}

TEST(TMultiPartTest, Success) {
    auto parts = MakeMultiPartBody({
        {"application", "json", "json_body"},
        {"message", "rfc822", "message_body"}
    });
    auto [json, msg] = ReadMultiPartBody(parts);
    ASSERT_EQ(std::string(json.begin(), json.end()), "json_body");
    ASSERT_EQ(std::string(msg.begin(), msg.end()), "message_body");
}

class TMultiPartTypesTest : public TestWithParam<TPartInfos> {
};

TEST_P(TMultiPartTypesTest, WrongTypesThrows) {
    auto parts = MakeMultiPartBody(GetParam());
    ASSERT_THROW(ReadMultiPartBody(parts), std::logic_error);
}

INSTANTIATE_TEST_SUITE_P(
    TMultiPartTests,
    TMultiPartTypesTest,
    ::testing::Values(
        TPartInfos{
            {"application", "", ""},
            {"message", "rfc822", ""}
        },
        TPartInfos{
            {"application", "json", ""},
            {"message", "", ""}
        },
        TPartInfos{
            {"text", "plain", ""},
            {"message", "rfc822", ""}
        },
        TPartInfos{
            {"application", "json", ""},
            {"text", "plain", ""}
        },
        TPartInfos{
            {"message", "rfc822", ""},
            {"application", "json", ""}
        }
    )
);

class TMultiPartCountTest : public TestWithParam<int> {
};

TEST_P(TMultiPartCountTest, WrongNumberOfPartsThrows) {
    std::vector<ymod_webserver::content_part> parts(GetParam());
    ASSERT_THROW(ReadMultiPartBody(parts), std::logic_error);
}

INSTANTIATE_TEST_SUITE_P(
    TMultiPartTests,
    TMultiPartCountTest,
    ::testing::Values(0, 1, 3));


TEST(ParseBodyTest, WrongNumberOfPartsThrows) {
    ASSERT_THROW(ParseBody(nullptr), std::runtime_error);
}

TEST(ParseBodyTest, Success) {
    auto request = boost::make_shared<ymod_webserver::request>();
    request->content.type = "multipart";
    request->content.subtype = "mixed";
    request->childs = MakeMultiPartBody({
        {"application", "json", R"({"key": "value"})"},
        {"message", "rfc822", "message_body"}
    });
    auto [json, msg] = ParseBody(request);
    ASSERT_EQ(json["key"], "value");
    ASSERT_EQ(std::string(msg.begin(), msg.end()), "message_body");
}

TEST(ParseBodyTest, BrokenJsonThrows) {
    auto request = boost::make_shared<ymod_webserver::request>();
    request->content.type = "multipart";
    request->content.subtype = "mixed";
    request->childs = MakeMultiPartBody({
        {"application", "json", "{\"key}"},
        {"message", "rfc822", "message_boxy"}
    });
    ASSERT_THROW(ParseBody(request), NJson::TJsonException);
}

TEST(ParseBodyTest, BadRequestContentTypeThrows) {
    auto request = boost::make_shared<ymod_webserver::request>();
    request->content.type = "application";
    ASSERT_THROW(ParseBody(request), std::logic_error);
}
