#include <mail/nwsmtp/src/xyandexhint/xyandexhint.h>

#include <library/cpp/testing/gtest_boost_extensions/extensions.h>

#include <gtest/gtest.h>

namespace {

using namespace NNwSmtp;
using namespace NNwSmtp::NHint;
using namespace testing;

TEST(xyandexhint, construct) {
    using SimpleXYandexHint = TXYandexHint<TFid, TLabels, THdrDate, TMixed, TSkipMetaMsg>;

    SimpleXYandexHint hint(TMixed{1234567}, TFid{"fid_id:ABCDE"}, THdrDate{101010});

    EXPECT_EQ(hint.GetTMixed().value.get(), 1234567U);
    EXPECT_EQ(hint.GetTFid().value, "fid_id:ABCDE");
    EXPECT_EQ(hint.GetTHdrDate().value.get(), 101010);

    hint = SimpleXYandexHint(TLabels{{"one", "two", "one", "three"}});
    EXPECT_EQ(hint.GetTLabels().value.size(), 3U);

    hint = SimpleXYandexHint(TSkipMetaMsg{true});
    EXPECT_EQ(hint.GetTSkipMetaMsg().value, true);
}

TEST(xyandexhint, constructCTAD) {
    auto hint = TXYandexHint{TMixed{1234567}, TFid{"fid_id:ABCDE"}, THdrDate{101010}};

    EXPECT_EQ(hint.GetTMixed().value.get(), 1234567U);
    EXPECT_EQ(hint.GetTFid().value, "fid_id:ABCDE");
    EXPECT_EQ(hint.GetTHdrDate().value.get(), 101010);
}

TEST(xyandexhint, set) {
    using SimpleXYandexHint = TXYandexHint<TUserLabels, TIpFrom, TNotify, TMid>;

    SimpleXYandexHint hint;

    hint.Set(TUserLabels{{"one", "two", "three", "one"}});
    hint.Set(TMid{"mid_id"});
    hint.Set(TNotify{true});
    hint.Set(TIpFrom{"127.0.0.1"});

    EXPECT_EQ(hint.GetTUserLabels().value.size(), 3U);
    EXPECT_EQ(hint.GetTIpFrom().value, "127.0.0.1");
    EXPECT_EQ(hint.GetTNotify().value.get(), true);
    EXPECT_EQ(hint.GetTMid().value, "mid_id");
}

TEST(xyandexhint, fillFromString) {
    const std::string paramsStr = "fid=fid_id\nlabel=firstLabel\nmixed=1\nlabel=secondLabel";
    auto encoded = yplatform::base64_encode(paramsStr.begin(), paramsStr.end());
    std::string encodedStr(encoded.begin(), encoded.end());

    using SimpleXYandexHint = TXYandexHint<TFid, TLabels, TMixed>;
    SimpleXYandexHint hint;

    hint.Fill(encodedStr);

    EXPECT_EQ(hint.GetTFid().value, "fid_id");
    EXPECT_EQ(hint.GetTMixed().value.get(), 1U);

    std::set<std::string> mylabels{"firstLabel", "secondLabel"};

    for (const auto & label : hint.GetTLabels().value) {
        ASSERT_NE(mylabels.find(label), mylabels.end());
        mylabels.erase(label);
    }

    EXPECT_TRUE(mylabels.empty());
}

TEST(xyandexhint, fillFromMap) {
    using SimpleXYandexHint = TXYandexHint< TSource, TFid, TImapLabels, TLabels, TSkipMetaMsg, TForward, TBcc>;
    SimpleXYandexHint hint;

    TParamsContainer params{
        {"source", "source1"},
        {"imaplabel", "imap1"},
        {"imaplabel", "imap2"},
        {"forward", "0"}};
    hint.Fill(params);

    params = TParamsContainer{
        {"fid", "fid_id"},
        {"imaplabel", "imap1"},
        {"imaplabel", "imap3"},
        {"imaplabel", "imap4"},
        {"bcc", "bcc1@ya.ru"}};
    hint.Fill(params);

    params = TParamsContainer{
        {"source", "source2"},
        {"bcc", "bcc2@ya.ru"},
        {"forward", "1"},
        {"skip_meta_msg", "0"}};
    hint.Fill(params);

    EXPECT_EQ(hint.GetTSource().value, "source2");
    EXPECT_EQ(hint.GetTFid().value, "fid_id");
    EXPECT_EQ(hint.GetTImapLabels().value.size(), 4U);
    EXPECT_TRUE(hint.GetTLabels().value.empty());
    EXPECT_EQ(hint.GetTSkipMetaMsg().value.get(), false);
    EXPECT_EQ(hint.GetTForward().value.get(), true);
    EXPECT_EQ(hint.GetTBcc().value.size(), 2U);
}

TEST(xyandexhint, write) {
    using SimpleXYandexHint = TXYandexHint<TSource, TFid, TImapLabels, TLabels, TSkipMetaMsg, TForward, TBcc>;
    SimpleXYandexHint hint;

    hint.Set(TSource{"sourceId"});
    hint.Set(TImapLabels{{"imap1", "imap2", "imap3"}});
    hint.Set(TSkipMetaMsg{true});
    hint.Set(TBcc{{"first@ya.ru", "second@ya.ru"}});

    std::ostringstream oss;
    oss << hint;

    auto hintStr = oss.str();
    hintStr = hintStr.substr(hintStr.find(' ') + 1);
    boost::trim(hintStr);

    SimpleXYandexHint extracted;
    extracted.Fill(hintStr);

    EXPECT_EQ(extracted.GetTSource().value, "sourceId");
    EXPECT_TRUE(extracted.GetTFid().value.empty());

    EXPECT_EQ(extracted.GetTImapLabels().value.size(), 3U);
    EXPECT_TRUE(extracted.GetTLabels().value.empty());

    ASSERT_TRUE(extracted.GetTSkipMetaMsg().value.is_initialized());
    EXPECT_EQ(extracted.GetTSkipMetaMsg().value.get(), true);
    ASSERT_FALSE(extracted.GetTForward().value.is_initialized());

    EXPECT_EQ(extracted.GetTBcc().value.size(), 2U);
}

TEST(xyandexhint, toString) {
    using SimpleXYandexHint = TXYandexHint<TEmail, TAllowAutoReply, TBcc>;
    SimpleXYandexHint hint;

    hint.Set(TBcc{{"first@ya.ru", "second@ya.ru"}});
    hint.Set(TEmail{"email@ya.ru"});

    // email=email@ya.ru\nbcc=first@ya.ru\nbcc=second@ya.ru\n
    EXPECT_EQ(hint.Str(), "X-Yandex-Hint: "
        "ZW1haWw9ZW1haWxAeWEucnUKYmNjPWZpcnN0QHlhLnJ1CmJjYz1zZWNvbmRAeWEucnUK\r\n");

    hint.Set(TAllowAutoReply{true});

    // email=email@ya.ru\nallow_autoreply=1\nbcc=first@ya.ru\nbcc=second@ya.ru\n
    EXPECT_EQ(hint.Str(), "X-Yandex-Hint: "
        "ZW1haWw9ZW1haWxAeWEucnUKYWxsb3dfYXV0b3JlcGx5PTEKYmNjPWZpcnN0QHlhLnJ1CmJjYz1zZWNvbmRAeWEucnUK\r\n");
}

TEST(xyandexhint, decodeEncodeViaRawParameters) {
    std::string header = "YT0xCmI9Mgo=";
    TParamsContainer params;
    ParseBase64(header, params);
    EXPECT_NE(params.find("a"), params.end());
    EXPECT_NE(params.find("b"), params.end());

    auto newHeader = MakeHintFromParams(params);
    EXPECT_EQ(newHeader, "X-Yandex-Hint: YT0xCmI9Mgo=\r\n");
}

} // namespace anonymous
