#include <mail/nwsmtp/src/yarm/client_impl.h>

#include "base_http_test.h"

#include <mail/nwsmtp/src/yarm/error_code.h>
#include <mail/nwsmtp/ut/init_log.h>
#include <mail/nwsmtp/ut/util/http_client_mock.h>

#include <gtest/gtest.h>

namespace {

using namespace NNwSmtp;
using namespace NNwSmtp::NTests;
using namespace NNwSmtp::NYarm;

struct TTestYarmClient: Test {
    void SetUp() override {
        NTesting::InitGlobalLog();
    }

    void AsyncRun(TCallback callback) {
        YarmClient->AsyncRun(Context, Request, std::move(callback));
        Io.run();
    }

    const Options::YarmOpts Options = Options::YarmOpts {
        .secret = "abcdefg",
    };
    const std::shared_ptr<StrictMock<TClusterClientMock>> ClusterClient = GetStrictMockedClusterClient();
    const TClientPtr YarmClient = std::make_shared<TClientImpl>(Options, ClusterClient, Io);
    const TContextPtr Context = boost::make_shared<TContext>("", "", "", "");

    const TRequest Request = TRequest {
        .Uid = "88005553535",
        .Email = "hello_darkness@my.old.friend",
    };
    const yhttp::request ClientRequest = yhttp::request::GET(
        "/api/v2/smtp_data?uid=88005553535&email=hello_darkness%40my.old.friend&secret=abcdefg"
    );

    boost::asio::io_service Io;
};

TEST_F(TTestYarmClient, ForClientErrorShouldPassTheError) {
    const TErrorCode clientError = ymod_httpclient::http_error::code::ssl_error;

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(clientError, yhttp::response {}));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, ymod_httpclient::http_error::code::ssl_error);
    });
}

TEST_F(TTestYarmClient, For503ShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 503,
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, ForBrokenJsonInResponseShouldReturnParseError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"broken":"json")",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::ParseError);
    });
}

TEST_F(TTestYarmClient, For500AndNoDataShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 500,
        .body = R"({})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, For200AndNoDataShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, For500AndErrorShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 500,
        .body = R"({"error":{"description":"get access token exception: No known oauth providers for server: smtp.mail.ru"}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, For200AndErrorShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"error":{"description":"get access token exception: No known oauth providers for server: smtp.mail.ru"}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, For500AndNoSuchCollectorShouldReturnNotFoundError) {
    const auto clientResponse = yhttp::response {
        .status = 500,
        .body = R"({"error":{"description":"no such collector"}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::NotFound);
    });
}

TEST_F(TTestYarmClient, ForIncompleteSmtpDataShouldReturnParseError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"smtp_data":{"login":"rpop.from.user.oauth@mail.ru"}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::ParseError);
    });
}

TEST_F(TTestYarmClient, ForEmptyLoginShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"smtp_data":{"login":"","pass":"MySuperPassw0rd","oauth":false}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, ForEmptyPasswordShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"smtp_data":{"login":"rpop.from.user.oauth@mail.ru","pass":"","oauth":false}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, ForEmptyLoginAndPasswordShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"smtp_data":{"login":"","pass":"","oauth":false}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, ForEmptyTokenShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"smtp_data":{"login":"rpop.from.user.oauth@mail.ru","pass":"","oauth":true}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, ForEmptyLoginAndTokenShouldReturnOtherError) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"smtp_data":{"login":"","pass":"","oauth":true}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, EError::OtherError);
    });
}

TEST_F(TTestYarmClient, ForGoodSmtpDataResponse) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"request":"smtp_data","host":"rpop4o.mail.yandex.net","id":"AOS9WcJAgmI1",)"
                R"("smtp_data":{"login":"rpop.from.user.oauth@mail.ru",)"
                R"("pass":"3bbe71990c787051905570181239b5920f25964437363830","oauth":true,)"
                R"("server":"smtp.mail.ru","port":"465","ssl":true}})",
    };

    InSequence seq;
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    AsyncRun([](auto errc, auto result) {
        ASSERT_FALSE(errc);
        EXPECT_EQ(result.Login, "rpop.from.user.oauth@mail.ru");
        EXPECT_EQ(result.PasswordOrToken, "3bbe71990c787051905570181239b5920f25964437363830");
        EXPECT_EQ(result.IsOauth, true);
    });
}

} // namespace anonymous
