import asyncio
from typing import Any, Optional

from aiohttp import web
from aiopg.sa.engine import Engine

from sendr_aiohttp import BaseUrlDispatcher, create_apispec, setup_swagger_route
from sendr_qlog.http.aiohttp import get_middleware_logging_adapter, signal_request_id_header

from mail.ohio.ohio import __version__
from mail.ohio.ohio.api.middlewares import middleware_response_formatter, middleware_stats, middleware_tvm
from mail.ohio.ohio.api.routes.customer import CUSTOMER_ROUTES
from mail.ohio.ohio.api.routes.internal import INTERNAL_ROUTES
from mail.ohio.ohio.api.routes.utility import UTILITY_ROUTES
from mail.ohio.ohio.api.routes.service import SERVICE_ROUTES
from mail.ohio.ohio.api.routes.services import SERVICES_ROUTES
from mail.ohio.ohio.conf import settings
from mail.ohio.ohio.interactions.base import BaseInteractionClient, create_connector
from mail.ohio.ohio.utils.db import create_configured_engine


class OhioApplication(web.Application):
    _urls = (
        CUSTOMER_ROUTES,
        INTERNAL_ROUTES,
        UTILITY_ROUTES,
        SERVICE_ROUTES,
        SERVICES_ROUTES
    )

    def __init__(self, db_engine: Optional[Engine] = None):
        super().__init__(
            router=BaseUrlDispatcher(),
            middlewares=(
                middleware_stats,
                get_middleware_logging_adapter(),
                middleware_response_formatter,
                middleware_tvm,
            ),
        )

        if db_engine:
            self.db_engine = db_engine
        else:
            self.on_startup.append(self.open_engine)
            self.on_cleanup.append(self.close_engine)

        self.add_routes()
        self.add_sentry()
        self.on_response_prepare.append(signal_request_id_header)
        self.on_startup.append(self.create_connector)
        self.on_cleanup.append(self.close_engine)
        self.on_cleanup.append(self.close_connector)

    def add_routes(self) -> None:
        for routes in self._urls:
            self.router.add_routes(routes)
        if settings.ENABLE_APISPEC:
            spec = create_apispec(
                app=self,
                title='ohio',
                version=__version__,
            )
            setup_swagger_route(
                app=self,
                spec=spec,
                swagger_route='/swagger.json',
                swagger_route_name='swagger',
            )

    def add_sentry(self) -> None:
        from sendr_qlog.sentry import sentry_init
        if settings.SENTRY_DSN:
            self.on_cleanup.append(sentry_init(settings.SENTRY_DSN, release=__version__))

    async def open_engine(self, _: Any) -> None:
        loop = asyncio.get_event_loop()
        self.db_engine = create_configured_engine(loop)

    async def close_engine(self, _: web.Application) -> None:
        if self.db_engine:
            self.db_engine.close()
        await self.db_engine.wait_closed()

    async def create_connector(self, _: Any) -> None:
        BaseInteractionClient.CONNECTOR = create_connector()

    async def close_connector(self, _: Any) -> None:
        await BaseInteractionClient.close_connector()
