from typing import Optional

from aiohttp import web

from sendr_aiohttp.handler import request_schema, response_schema

from mail.ohio.ohio.api.exceptions import TVMUserTicketException
from mail.ohio.ohio.api.handlers.base import BaseHandler
from mail.ohio.ohio.api.schemas.customer import (
    get_customer_order_by_id_response_schema, get_customer_orders_query_schema, get_customer_orders_response_schema,
    get_customer_services_response_schema
)
from mail.ohio.ohio.api.schemas.path import uid_order_id_path_schema, uid_path_schema
from mail.ohio.ohio.core.actions.order.get_by_id import GetByIdOrderAction
from mail.ohio.ohio.core.actions.order.get_for_customer import GetForCustomerOrderAction
from mail.ohio.ohio.core.actions.order.get_services import GetServicesOrderAction
from mail.ohio.ohio.api.handlers.base import is_globally_disabled


class BaseCustomerHandler(BaseHandler):
    """
    Base handler for routes containing `customer_uid`.
    """

    def _check_user_ticket(self) -> None:
        customer_uid: str = self.request.match_info['customer_uid']
        default_uid: Optional[int] = self.tvm.default_uid
        if default_uid is None or str(default_uid) != customer_uid:
            raise TVMUserTicketException

    async def _iter(self):
        self._check_user_ticket()
        return await super()._iter()


class CustomerOrdersHandler(BaseCustomerHandler):
    @request_schema(uid_path_schema, location='match_info')
    @request_schema(get_customer_orders_query_schema, location='query')
    @response_schema(get_customer_orders_response_schema)
    async def get(self) -> web.Response:
        data = await self.get_data()
        orders, next_keyset = await self.run_action(GetForCustomerOrderAction, **data)
        if is_globally_disabled():
            orders=[]
        return self.make_response({
            'data': {
                'orders': orders,
                'next': next_keyset,
            }
        })


class CustomerOrderByIdHandler(BaseCustomerHandler):
    @request_schema(uid_order_id_path_schema, location='match_info')
    @response_schema(get_customer_order_by_id_response_schema)
    async def get(self) -> web.Response:
        data = await self.get_data()
        order = await self.run_action(GetByIdOrderAction, **data)
        if is_globally_disabled():
            order = ""
        return self.make_response({
            'data': {
                'order': order,
            },
        })


class CustomerServicesHandler(BaseCustomerHandler):
    @request_schema(uid_path_schema, location='match_info')
    @response_schema(get_customer_services_response_schema)
    async def get(self) -> web.Response:
        data = await self.get_data()
        services = await self.run_action(GetServicesOrderAction, **data)
        if is_globally_disabled():
            services=[]
        return self.make_response({
            'data': {
                'services': services,
            },
        })
