from dataclasses import fields
from typing import Optional, Tuple

from mail.ohio.ohio.core.actions.base import BaseDBAction
from mail.ohio.ohio.core.entities.order import Order

MISSING = object()


class BaseWriteOrderAction(BaseDBAction):
    """
    Must be used when creating / updating order.
    """

    async def _handle(self) -> Tuple[Optional[Order], Optional[Order]]:
        """
        Must return tuple containing old and new order entities. Must not create / update order in storage.  In case
        old order does not exist must return None in it's place. In case order was not created / updated must return
        None in it's place.
        """
        raise NotImplementedError

    async def handle(self) -> Optional[Order]:
        with self.logger:
            async with self.storage.conn.begin():
                old_order, new_order = await self._handle()
                self.logger.context_push(
                    order_id=getattr(old_order, 'order_id', None) or getattr(new_order, 'order_id', None),
                )

                # Order was not created
                if new_order is None:
                    self.logger.info('New order is empty. Nothing to do.')
                    return None

                # Building diff
                diff = {}
                for field in fields(new_order):
                    old_value = getattr(old_order, field.name, MISSING)
                    new_value = getattr(new_order, field.name)
                    if old_value != new_value:
                        diff[field.name] = new_value

                if not diff:
                    self.logger.info('Order did not change. Not writing anything.')
                    return new_order

                # Writing order
                if old_order is None:
                    new_order = await self.storage.order.create(new_order)
                    self.logger.context_push(order_id=new_order.order_id)
                    self.logger.context_push(uid=new_order.customer_uid)
                    self.logger.info('Order created.')
                else:
                    new_order = await self.storage.order.save(new_order)
                    self.logger.info('Order updated.')

                return new_order
