from copy import deepcopy
from datetime import datetime
from typing import Optional, Tuple

from mail.ohio.ohio.core.actions.order.base import BaseWriteOrderAction
from mail.ohio.ohio.core.entities.customer import Customer
from mail.ohio.ohio.core.entities.order import Order, OrderData, OrderStatus
from mail.ohio.ohio.storage.exceptions import (
    CustomerNotFoundStorageError, OrderNotFoundStorageError, ServiceNotFoundStorageError
)


class CreateOrUpdateFromTrustOrderAction(BaseWriteOrderAction):
    transact = True

    def __init__(self,
                 customer_uid: int,
                 service_id: int,
                 payments_service_id: int,
                 trust_purchase_token: str,
                 trust_payment_id: str,
                 merchant_uid: int,
                 service_merchant_id: int,
                 payments_order_id: int,
                 status: OrderStatus,
                 order_data: OrderData,
                 service_data: dict,
                 revision: int,
                 created: datetime,
                 subservice_id: Optional[str] = None,
                 ):
        super().__init__()
        self._customer_uid = customer_uid
        self._service_id = service_id
        self._payments_service_id = payments_service_id
        self._trust_purchase_token = trust_purchase_token
        self._merchant_uid = merchant_uid
        self._service_merchant_id = service_merchant_id
        self._payments_order_id = payments_order_id
        self._status = status
        self._order_data = order_data
        self._service_data = service_data
        self._revision = revision
        self._created = created
        self._subservice_id = subservice_id
        self._trust_payment_id = trust_payment_id

    def _update_order_entity(self, order: Order) -> None:
        order.subservice_id = self._subservice_id
        order.merchant_uid = self._merchant_uid
        order.service_merchant_id = self._service_merchant_id
        order.payments_order_id = self._payments_order_id
        order.trust_payment_id = self._trust_payment_id
        order.status = self._status
        order.order_data = self._order_data
        order.service_data = self._service_data
        order.order_revision = order.service_revision = self._revision
        order.created = self._created
        self.logger.info('Order entity updated.')

    async def _handle(self) -> Tuple[Optional[Order], Optional[Order]]:
        with self.logger:
            self.logger.context_push(
                customer_uid=self._customer_uid,
                trust_service_id=self._service_id,
                payments_service_id=self._payments_service_id,
                merchant_uid=self._merchant_uid,
                payments_order_id=self._payments_order_id,
                trust_payment_id=self._trust_payment_id,
                trust_purchase_token=self._trust_purchase_token,
            )

            # Creating customer
            try:
                await self.storage.customer.get(self._customer_uid)
                self.logger.info('Customer exists.')
            except CustomerNotFoundStorageError:
                await self.storage.customer.create(Customer(customer_uid=self._customer_uid))
                self.logger.info('Customer created.')

            # Getting service
            try:
                service = await self.storage.service.get(self._service_id)
            except ServiceNotFoundStorageError:
                self.logger.warning('Service not found.')
                return None, None
            if not service.enabled:
                self.logger.warning('Serivce not enabled.')
                return None, None
            assert service.service_id is not None
            self.logger.info('Service found.')

            # Getting order
            old_order: Optional[Order]
            try:
                order = await self.storage.order.get_by_trust_purchase_token(
                    customer_uid=self._customer_uid,
                    trust_purchase_token=self._trust_purchase_token,
                )
                old_order = deepcopy(order)
                assert order.service_id == self._service_id  # sanity check
                self.logger.context_push(order_id=order.order_id)
                self.logger.info('Order exists. Updating...')
            except OrderNotFoundStorageError:
                order = Order(
                    customer_uid=self._customer_uid,
                    service_id=self._service_id,
                    trust_purchase_token=self._trust_purchase_token,
                )
                old_order = None
                self.logger.info('Order does not exist. Creating...')

            # Setting order attributes
            self._update_order_entity(order)

            return old_order, order
