from datetime import datetime
from typing import Dict, List, Optional, Tuple

from sendr_utils import alist

from mail.ohio.ohio.core.actions.base import BaseDBAction
from mail.ohio.ohio.core.entities.order import Order


class GetYandexAccountForCustomerOrderAction(BaseDBAction):
    """
    Returns list of orders with yandex account operations and arguments for the next call.
    """

    def __init__(self,
                 customer_uid: int,
                 created_keyset: Optional[datetime] = None,
                 order_id_keyset: Optional[int] = None,
                 limit: Optional[int] = None,
                 ):
        super().__init__()
        self._customer_uid = customer_uid
        self._created_keyset = created_keyset
        self._order_id_keyset = order_id_keyset
        self._limit = limit

    async def handle(self) -> Tuple[List[Order], Dict[str, int]]:
        orders = await alist(
            self.storage.order.find_yandex_account_for_customer(
                customer_uid=self._customer_uid,
                created_keyset=self._created_keyset,
                order_id_keyset=self._order_id_keyset,
                limit=self._limit,
            )
        )
        next_keyset = {} if not orders else {
            'created_keyset': orders[-1].created,
            'order_id_keyset': orders[-1].order_id,
        }
        self.logger.info(
            f'GetYandexAccountForCustomerOrderAction: uid: {self._customer_uid}, '
            f'created_keyset: {self._created_keyset}, '
            f'order_id_keyset: {self._order_id_keyset}, '
            f'limit" {self._limit}, returning: {orders.__repr__()}')
        return orders, next_keyset
