from sendr_aiopg import BaseMapper, CRUDQueries
from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_utils import utcnow

from mail.ohio.ohio.core.entities.service import Service
from mail.ohio.ohio.storage.db.tables import services as t_services
from mail.ohio.ohio.storage.exceptions import ServiceNotFoundStorageError
from typing import AsyncIterable
from mail.ohio.ohio.core.entities.order_service import OrderService
from sqlalchemy import select


class ServiceDataMapper(SelectableDataMapper):
    selectable = t_services
    entity_class = Service


class ServiceDataDumper(TableDataDumper):
    table = t_services
    entity_class = Service


class ServiceMapper(BaseMapper):
    _builder = CRUDQueries(
        base=t_services,
        id_fields=('service_id',),
        mapper_cls=ServiceDataMapper,
        dumper_cls=ServiceDataDumper,
    )

    async def create(self, obj: Service) -> Service:
        obj.created = obj.updated = utcnow()
        query, mapper = self._builder.insert(obj, ignore_fields=('service_id',))
        return mapper(await self._query_one(query))

    async def get(self, service_id: int, for_update: bool = False) -> Service:
        query, mapper = self._builder.select(id_values=(service_id,), for_update=for_update)
        return mapper(await self._query_one(query, raise_=ServiceNotFoundStorageError))

    async def get_all(self) -> AsyncIterable[OrderService]:
        query = select([t_services.c.service_id, t_services.c.trust_service_id])
        async for row in self._query(query):
            yield OrderService(service_id=row['service_id'], subservice_id=row['trust_service_id'])

    async def get_by_payments_service_id(self, payments_service_id: int, for_update: bool = False) -> Service:
        query, mapper = self._builder.select(
            filters={'payments_service_id': payments_service_id},
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=ServiceNotFoundStorageError))

    async def get_by_trust_service_id(self, trust_service_id: int, for_update: bool = False) -> Service:
        query, mapper = self._builder.select(
            filters={'trust_service_id': trust_service_id},
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=ServiceNotFoundStorageError))

    async def save(self, obj: Service) -> Service:
        obj.updated = utcnow()
        query, mapper = self._builder.update(obj, ignore_fields=('created', 'service_type'))
        return mapper(await self._query_one(query, raise_=ServiceNotFoundStorageError))
