package ru.yandex.ohio.backend;

import java.nio.charset.CharacterCodingException;
import java.util.Objects;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.FakeAsyncConsumer;
import ru.yandex.http.util.nio.StatusCheckFallbackAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;

public abstract class FnsListingHandlerBase
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final DateTimeFormatter INPUT_DATE_FORMAT =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss'Z'").withZoneUTC();
    private static final DateTimeFormatter DATE_TIME_FORMATTER =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZZ").withZoneUTC();
    private static final DateTimeFormatter UPDATED_TIME_FORMATTER =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZZ").withZoneUTC();

    protected final OhioBackend server;

    protected FnsListingHandlerBase(final OhioBackend server) {
        this.server = server;
    }

    @Override
    public FakeAsyncConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new FakeAsyncConsumer<>(request);
    }

    protected void handle(final FnsListingContext listingContext)
        throws HttpException
    {
        if (listingContext.limit() == 0) {
            new ResultPrinter(
                listingContext.session(),
                listingContext.jsonType())
                .completed(ListingHandlerBase.EMPTY_RESULT);
        } else {
            int limit = listingContext.limit();
            if (listingContext.startUuid() != null) {
                limit = ListingHandlerBase.increaseLimit(limit, 1);
            }
            listing(listingContext, listingContext.startOffset(), limit);
        }
    }

    private static long parseTimestamp(final String timestamp) {
        try {
            return DATE_TIME_FORMATTER.parseMillis(timestamp);
        } catch (RuntimeException e) {
            return INPUT_DATE_FORMAT.parseMillis(timestamp);
        }
    }

    private void listing(
        final FnsListingContext listingContext,
        final int offset,
        final int limit)
    {
        ProxySession session = listingContext.session();
        session.logger().info(
            "Trying to list " + limit + " documents with "
            + listingContext.limit()
            + " documents required starting from offset " + offset);
        long uid = listingContext.uid();
        AsyncClient client = server.dyngoClient().adjust(session.context());
        client.execute(
            server.dyngoHost(),
            new BasicAsyncRequestProducerGenerator(
                "/ya/user/" + uid
                + "/checks?sort=check_date+desc,uuid+desc&limit=" + limit
                + "&offset=" + offset),
            new StatusCheckFallbackAsyncResponseConsumerFactory<>(
                x -> x == HttpStatus.SC_NO_CONTENT,
                JsonAsyncTypesafeDomConsumerFactory.ANY_GOOD,
                null),
            session.listener().createContextGeneratorFor(client),
            new Callback(listingContext, offset, limit));
    }

    private class Callback extends AbstractProxySessionCallback<JsonObject> {
        private final FnsListingContext listingContext;
        private final int offset;
        private final int limit;
        private final PrefixedLogger logger;
        private String lastUuid;

        Callback(
            final FnsListingContext listingContext,
            final int offset,
            final int limit)
        {
            super(listingContext.session());
            this.listingContext = listingContext;
            this.offset = offset;
            this.limit = limit;
            logger = session.logger();
        }

        @Override
        public void completed(final JsonObject result) {
            if (result == null) {
                new ResultPrinter(
                    listingContext.session(),
                    listingContext.jsonType())
                    .completed(ListingHandlerBase.EMPTY_RESULT);
                return;
            }
            try {
                completed(result.asList());
            } catch (CharacterCodingException | JsonException e) {
                failed(
                    new ServiceUnavailableException(
                        "Failed to parse receipts listing: "
                        + JsonType.NORMAL.toString(result),
                        e));
            }
        }

        private JsonMap convertDoc(
            final JsonMap doc,
            final int pos)
            throws BadRequestException, JsonException
        {
            lastUuid = doc.getString("uuid");

            JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);
            JsonMap representation =
                Objects.requireNonNullElse(
                    doc.get("representation").asMapOrNull(),
                    JsonMap.EMPTY);
            long checkDate = parseTimestamp(doc.getString("check_date"));
            long createdAt = parseTimestamp(doc.getString("created_at"));
            if (checkDate > createdAt) {
                logger.info(
                    "Fallback from check_date " + checkDate
                    + " to created_at " + createdAt);
                checkDate = createdAt;
            }
            JsonString created =
                new JsonString(DATE_TIME_FORMATTER.print(checkDate));
            String receiptUrl = representation.getString("html", "");
            if (!receiptUrl.isEmpty()) {
                result.put("receipt_url", new JsonString(receiptUrl));
                result.put("receipt_timestamp", created);
            }
            result.put("trust_purchase_token", new JsonString(lastUuid));
            String affiliateId = doc.getString("affiliate_id", null);
            if (affiliateId != null) {
                result.put("subservice_id", new JsonString(affiliateId));
            }
            result.put("created", created);
            result.put(
                "updated",
                new JsonString(
                    UPDATED_TIME_FORMATTER.print(
                        parseTimestamp(doc.getString("updated_at")))));
            result.put(
                "service_merchant_id",
                new JsonLong(listingContext.uid()));
            result.put("payments_order_id", JsonLong.ONE);

            JsonMap serviceData = new JsonMap(BasicContainerFactory.INSTANCE);
            serviceData.put("user_account", JsonString.EMPTY);
            // XXX
            serviceData.put("payment_method", new JsonString("unknown"));
            serviceData.put("trust_group_id", JsonString.EMPTY);
            serviceData.put("cashback_amount", JsonLong.ZERO);
            serviceData.put("trust_payment_id", new JsonString(lastUuid));
            serviceData.put("composite_payment_id", JsonLong.ZERO);
            result.put("service_data", serviceData);
            result.put("customer_uid", new JsonLong(listingContext.uid()));
            result.put("service_revision", JsonLong.ONE);
            result.put("trust_payment_id", new JsonString(lastUuid));
            result.put("order_id", new JsonLong(offset + pos));
            result.put("order_revision", JsonLong.ONE);
            // XXX
            result.put("status", new JsonString("paid"));
            result.put("merchant_uid", JsonLong.ONE);
            result.put("description", JsonString.EMPTY);
            // XXX
            result.put("currency", new JsonString("RUB"));
            JsonString total =
                new JsonString(
                    Double.toString(doc.getLong("check_sum") / 100d));
            result.put("total", total);
            result.put("refunds", JsonList.EMPTY);

            JsonMap retailer =
                Objects.requireNonNullElse(
                    doc.get("retailer").asMapOrNull(),
                    JsonMap.EMPTY);
            String retailerName = retailer.get("name").asStringOrNull();
            if (retailerName != null && !retailerName.isEmpty()) {
                result.put("retailer_name", new JsonString(retailerName));
            }
            String icon = retailer.get("icon").asStringOrNull();
            if (icon != null && !icon.isEmpty()) {
                result.put("retailer_icon", new JsonString(icon + "?res=s"));
            }

            JsonMap cashback = doc.get("cashback").asMapOrNull();
            if (cashback != null) {
                String type = cashback.getString("type", null);
                if (type != null) {
                    JsonMap cashbackInfo =
                        new JsonMap(BasicContainerFactory.INSTANCE);
                    cashbackInfo.put("type", new JsonString(type));
                    long amount = cashback.getLong("amount", 0L);
                    cashbackInfo.put(
                        "amount",
                        new JsonString(Double.toString(amount / 100d)));
                    result.put("cashback", cashbackInfo);
                }
            }
            JsonMap features = doc.get("features").asMapOrNull();
            if (features != null) {
                result.put("features", features);
            }

            JsonList receiptItems = doc.get("items").asListOrNull();
            JsonList items;
            if (receiptItems == null) {
                JsonMap item = new JsonMap(BasicContainerFactory.INSTANCE);
                item.put("name", new JsonString("none"));
                item.put("image_path", JsonNull.INSTANCE);
                // XXX
                item.put("currency", new JsonString("RUB"));
                item.put("nds", JsonString.EMPTY);
                item.put("price", total);
                item.put("amount", total);
                items = new JsonList(BasicContainerFactory.INSTANCE, 1);
                items.add(item);
            } else {
                int itemsSize = receiptItems.size();
                items = new JsonList(BasicContainerFactory.INSTANCE, itemsSize);
                int transportingIdx = -1;
                int extendedSubscriptionIdx = -1;
                for (int i = 0; i < itemsSize; ++i) {
                    JsonMap row = receiptItems.get(i).asMap();
                    JsonMap item = new JsonMap(BasicContainerFactory.INSTANCE);
                    String name = row.getString("name", "");
                    if ("none".equals(name)) {
                        name = "";
                    }
                    ItemAlias alias =
                        ListingHandlerBase.detectAlias(
                            name,
                            0L,
                            JsonMap.EMPTY);
                    String nameReplacement;
                    if (alias == null) {
                        nameReplacement = null;
                    } else {
                        nameReplacement = alias.nameReplacement();
                    }
                    if (nameReplacement != null) {
                        name = nameReplacement;
                    } else if (name.isEmpty()) {
                        name = "none";
                    }
                    item.put("name", new JsonString(name));
                    if (alias == ItemAlias.TRANSPORTING_FEE) {
                        transportingIdx = i;
                    } else if (alias == ItemAlias.EXTENDED_SUBSCRIPTION) {
                        extendedSubscriptionIdx = i;
                    }
                    if (alias != null && alias.export()) {
                        item.put("alias", new JsonString(alias.aliasName()));
                    }
                    item.put("image_path", JsonNull.INSTANCE);
                    // XXX
                    item.put("currency", new JsonString("RUB"));
                    // XXX
                    item.put("nds", JsonString.EMPTY);
                    item.put(
                        "price",
                        new JsonString(
                            Double.toString(row.getLong("price") / 100d)));
                    item.put(
                        "amount",
                        new JsonString(
                            Double.toString(row.getLong("sum") / 100d)));
                    items.add(item);
                }
                ListingHandlerBase.foldExtendedSubscription(
                    items,
                    extendedSubscriptionIdx,
                    transportingIdx);
            }
            result.put("items", items);
            return result;
        }

        private void completed(final JsonList result)
            throws CharacterCodingException, JsonException
        {
            int size = result.size();
            int offset = 0;
            String startUuid = listingContext.startUuid();
            if (startUuid != null) {
                for (int i = 0; i < size; ++i) {
                    JsonMap receipt = result.get(i).asMap();
                    if (startUuid.equals(receipt.get("uuid").asString())) {
                        offset = i + 1;
                        break;
                    }
                }
            }
            int actualSize = Math.min(size - offset, listingContext.limit());
            int actualOffset = this.offset + offset;
            logger.info(
                size + " receipts received, internal offset is " + offset
                + ", actual size: " + actualSize);
            if (size >= limit && actualSize < listingContext.limit()) {
                logger.info("Requesting more data");
                listing(
                    listingContext,
                    actualOffset,
                    ListingHandlerBase.increaseLimit(limit, 1));
                return;
            }
            if (actualSize <= 0) {
                new ResultPrinter(
                    listingContext.session(),
                    listingContext.jsonType())
                    .completed(ListingHandlerBase.EMPTY_RESULT);
                return;
            }
            JsonList orders =
                new JsonList(BasicContainerFactory.INSTANCE, actualSize);
            for (int i = 0; i < actualSize; ++i) {
                JsonObject doc = result.get(i + offset);
                try {
                    orders.add(convertDoc(doc.asMap(), i));
                } catch (Exception e) {
                    listingContext.session().logger().log(
                        Level.WARNING,
                        "Failed to parse receipt: "
                        + JsonType.NORMAL.toString(doc),
                        e);
                }
            }

            JsonMap next = new JsonMap(BasicContainerFactory.INSTANCE, 3);
            next.put(
                "order_id_keyset",
                new JsonLong(actualOffset + actualSize));
            next.put("created_keyset", new JsonString(lastUuid));

            JsonMap data = new JsonMap(BasicContainerFactory.INSTANCE, 3);
            data.put("orders", orders);
            data.put("next", next);

            new ResultPrinter(
                listingContext.session(),
                listingContext.jsonType())
                .completed(data);
        }
    }
}

