package ru.yandex.ohio.backend;

import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.Cancellable;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.IntPair;
import ru.yandex.compress.GzipInputStream;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.EmptyCancellationSubscriber;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.FakeAsyncConsumer;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.NullObjectFilter;
import ru.yandex.json.dom.OrderingContainerFactory;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.util.timesource.TimeSource;

public class ImportRefundsHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private final OhioBackend server;

    public ImportRefundsHandler(final OhioBackend server) {
        this.server = server;
    }

    @Override
    public FakeAsyncConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new FakeAsyncConsumer<>(request);
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        new Context(server, new BasicProxySession(server, exchange, context))
            .processNextRecord();
    }

    private static class Context implements Cancellable {
        private final OhioBackend server;
        private final ProxySession session;
        private final BufferedReader reader;
        private volatile RecordInfo currentRecord = null;
        private volatile int linesProcessed;

        Context(final OhioBackend server, final ProxySession session)
            throws HttpException
        {
            this.server = server;
            this.session = session;
            File file = session.params().getInputFile("file");
            boolean ungzip = session.params().getBoolean("ungzip", false);
            linesProcessed = session.params().getInt("offset", 0);
            try {
                reader =
                    new BufferedReader(
                        new InputStreamReader(
                            openFile(file, ungzip),
                            StandardCharsets.UTF_8),
                        1 << 20);
                try {
                    for (int i = 0; i < linesProcessed; ++i) {
                        if (reader.readLine() == null) {
                            throw new EOFException(
                                "Unexpected EOF after line " + i);
                        }
                    }
                } catch (IOException e) {
                    try {
                        reader.close();
                    } catch (IOException ex) {
                        e.addSuppressed(ex);
                    }
                    throw new ServiceUnavailableException(e);
                }
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            }
            session.subscribeForCancellation(this);
        }

        private static InputStream openFile(
            final File file,
            final boolean ungzip)
            throws IOException
        {
            InputStream in = new FileInputStream(file);
            if (ungzip) {
                try {
                    return new GzipInputStream(in, 1 << 20);
                } catch (IOException e) {
                    try {
                        in.close();
                    } catch (IOException ex) {
                        e.addSuppressed(ex);
                    }
                    throw e;
                }
            } else {
                return in;
            }
        }

        @Override
        public boolean cancel() {
            session.logger().info("Closing reader");
            try {
                reader.close();
            } catch (IOException e) {
                session.logger().log(Level.WARNING, "Reader close failed", e);
            }
            return true;
        }

        public void processNextRecord() {
            currentRecord = null;
            try {
                while (currentRecord == null) {
                    String line = reader.readLine();
                    if (line == null) {
                        break;
                    }
                    try {
                        currentRecord = parseRecord(line);
                    } catch (JsonException e) {
                        session.logger().log(
                            Level.WARNING,
                            "Unable to parse line " + linesProcessed
                            + ':' + ' ' + line,
                            e);
                        ++linesProcessed;
                    }
                    if (currentRecord != null
                        && currentRecord.serviceOrderIdNumbers.isEmpty())
                    {
                        session.logger().info(
                            "No `service_order_id_number`s found in line "
                            + linesProcessed);
                        currentRecord = null;
                        ++linesProcessed;
                    }
                }
            } catch (IOException e) {
                session.handleException(
                    new ServiceUnavailableException(
                        "Read failed at line " + linesProcessed,
                        e));
                return;
            }
            session.logger().info(
                "Line read from position " + linesProcessed);
            if (currentRecord == null) {
                session.response(
                    HttpStatus.SC_OK,
                    linesProcessed + " lines processed");
                return;
            }

            importRecord();
        }

        private void importRecord() {
            server.sequentialRequest(
                EmptyCancellationSubscriber.INSTANCE,
                new PlainUniversalSearchProxyRequestContext(
                    new User("ohio_index", new LongPrefix(currentRecord.uid)),
                    null,
                    true,
                    server.searchClient(),
                    session.logger()),
                new BasicAsyncRequestProducerGenerator(
                    "/search?service=ohio_index&get=purchase_token,rows"
                    + "&prefix=" + currentRecord.uid + "&text=service_id:"
                    + currentRecord.serviceId + "+AND+NOT+has_refunds:1"),
                null,
                true,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                server.searchClient().httpClientContextGenerator(),
                new IdempotentFutureCallback<>(new RowsCallback(this)),
                server.producerClient(),
                server.producerClient().httpClientContextGenerator());
        }

        private void updateWithRefunds(final String purchaseToken)
            throws HttpException, IOException
        {
            QueryConstructor query = new QueryConstructor(
                "/update?import-refunds&service=ohio_index");
            query.append("prefix", currentRecord.uid);
            query.append("service_id", currentRecord.serviceId);
            query.append("purchase_token", purchaseToken);
            query.append(
                "service_order_id_numbers",
                currentRecord.serviceOrderIdNumbers.toString());

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(currentRecord.uid);
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writer.key("purchase_token");
                writer.value(purchaseToken);
                writer.key("data_format_version");
                writer.value(3L);
                writer.key("order_revision");
                writer.startObject();
                writer.key("function");
                writer.value("inc");
                writer.endObject();
                writer.key("update_timestamp");
                writer.value(TimeSource.INSTANCE.currentTimeMillis());
                JsonObject rows =
                    currentRecord.rows.filter(
                        NullObjectFilter.INSTANCE,
                        OrderingContainerFactory.INSTANCE);
                JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE, 4);
                map.put("rows", rows);
                map.put("amount", currentRecord.amount);
                map.put("currency", currentRecord.currency);
                JsonList list =
                    new JsonList(BasicContainerFactory.INSTANCE, 1);
                list.add(map);
                writer.key("refunds");
                writer.value(JsonType.NORMAL.toString(list));
                writer.endObject();
                writer.endArray();
                writer.endObject();
            }
            server.producerStoreClient().execute(
                server.producerStoreHost(),
                new BasicAsyncRequestProducerGenerator(
                    query.toString(),
                    sbw.toString()),
                StatusCodeAsyncConsumerFactory.ANY_GOOD,
                new IdempotentFutureCallback<>(
                    new UpdateCallback(this, purchaseToken)));
        }
    }

    private static RecordInfo parseRecord(final String record)
        throws IOException, JsonException
    {
        JsonMap map = TypesafeValueContentHandler.parse(record).asMap();
        long uid = map.get("creator_uid").asLong();
        long serviceId = map.get("service_id").asLong();
        JsonList rows =
            map.get(
                "payment_rows",
                TypesafeValueContentHandler::parse)
                .asList();
        int size = rows.size();
        Set<Long> serviceOrderIdNumbers = new HashSet<>(size << 1);
        for (int i = 0; i < size; ++i) {
            serviceOrderIdNumbers.add(
                rows.get(i)
                    .get("order")
                    .get("service_order_id_number")
                    .asLong());
        }
        return new RecordInfo(
            rows,
            uid,
            serviceId,
            map.get("amount"),
            map.get("currency"),
            serviceOrderIdNumbers);
    }

    private static class RecordInfo {
        private final JsonList rows;
        private final long uid;
        private final long serviceId;
        private final JsonObject amount;
        private final JsonObject currency;
        private final Set<Long> serviceOrderIdNumbers;

        RecordInfo(
            final JsonList rows,
            final long uid,
            final long serviceId,
            final JsonObject amount,
            final JsonObject currency,
            final Set<Long> serviceOrderIdNumbers)
        {
            this.rows = rows;
            this.uid = uid;
            this.serviceId = serviceId;
            this.amount = amount;
            this.currency = currency;
            this.serviceOrderIdNumbers = serviceOrderIdNumbers;
        }

        @Override
        public String toString() {
            return "RecordInfo(" + uid + ',' + serviceId + ','
                + serviceOrderIdNumbers + ')';
        }
    }

    private static class RowsCallback implements FutureCallback<JsonObject> {
        private final Context context;

        RowsCallback(final Context context) {
            this.context = context;
        }

        @Override
        public void cancelled() {
            context.session.logger().warning(
                "Request cancelled, total lines processed so far: "
                + context.linesProcessed);
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                Set<Long> serviceOrderIdNumbers =
                    context.currentRecord.serviceOrderIdNumbers;
                JsonList hits = result.get("hitsArray").asList();
                int hitsSize = hits.size();
                String purchaseToken = null;
                for (int i = 0; i < hitsSize && purchaseToken == null; ++i) {
                    JsonMap hit = hits.get(i).asMap();
                    JsonList rows =
                        Objects.requireNonNullElse(
                            hit.get(
                                "rows",
                                JsonNull.INSTANCE,
                                TypesafeValueContentHandler::parse)
                                .asListOrNull(),
                            JsonList.EMPTY);
                    int rowsSize = rows.size();
                    for (int j = 0; j < rowsSize; ++j) {
                        Long id =
                            Objects.requireNonNullElse(
                                rows.get(j).get("order").asMapOrNull(),
                                JsonMap.EMPTY)
                                .get("service_order_id_number")
                                .asLongOrNull();
                        if (serviceOrderIdNumbers.contains(id)) {
                            purchaseToken =
                                hit.get("purchase_token").asString();
                            break;
                        }
                    }
                }
                if (purchaseToken == null) {
                    context.session.logger().info(
                        "No records found for line " + context.linesProcessed);
                    ++context.linesProcessed;
                    context.processNextRecord();
                } else {
                    context.updateWithRefunds(purchaseToken);
                }
            } catch (HttpException | IOException | JsonException e) {
                context.session.logger().log(
                    Level.WARNING,
                    "Malformed search backend response for line "
                    + context.linesProcessed + ':' + ' '
                    + JsonType.NORMAL.toString(result));
                ++context.linesProcessed;
                context.processNextRecord();
            }
        }

        @Override
        public void failed(final Exception e) {
            context.session.logger().log(
                Level.WARNING,
                "Failed to get rows for line " + context.linesProcessed,
                e);
            context.importRecord();
        }
    }

    private static class UpdateCallback
        implements FutureCallback<IntPair<Void>>
    {
        private final Context context;
        private final String purchaseToken;

        UpdateCallback(final Context context, final String purchaseToken) {
            this.context = context;
            this.purchaseToken = purchaseToken;
        }

        @Override
        public void cancelled() {
            context.session.logger().warning(
                "Request cancelled, total lines processed so far: "
                + context.linesProcessed);
        }

        @Override
        public void completed(final IntPair<Void> result) {
            ++context.linesProcessed;
            context.session.logger().info(
                "Updated purchase_token " + purchaseToken
                + ", total lines processed: " + context.linesProcessed);
            context.processNextRecord();
        }

        @Override
        public void failed(final Exception e) {
            context.session.logger().log(
                Level.WARNING,
                "Failed to update with line " + context.linesProcessed
                + ':' + ' ' + context.currentRecord,
                e);
            try {
                context.updateWithRefunds(purchaseToken);
            } catch (HttpException | IOException ex) {
                context.session.logger().log(
                    Level.WARNING,
                    "Can't update purchase_token " + purchaseToken
                    + " at line " + context.linesProcessed,
                    ex);
                ++context.linesProcessed;
                context.processNextRecord();
            }
        }
    }
}

