package ru.yandex.ohio.backend;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.parser.uri.QueryConstructor;

public class ListingContext {
    protected OhioBackend server;
    protected ProxySession session;
    protected long uid;
    protected boolean hideFamilyPayments;
    protected List<Long> initiatorsUids;
    protected TerminalsInfos terminalsInfos;
    protected int limit;
    protected JsonType jsonType;
    protected Set<Long> serviceIds;
    protected boolean showYandexPay;
    protected long afterTimestamp;
    protected long beforeTimestamp;
    protected Long startPosTimestamp;
    protected String startPosPurchaseToken;

    public ListingContext(
        final OhioBackend server,
        final ProxySession session,
        final long uid)
        throws BadRequestException
    {
        this.server = server;
        this.session = session;
        this.uid = uid;
        hideFamilyPayments =
            session.params().getBoolean("hide-family-payments", false);
        initiatorsUids = session.params().getAll(
            "initiator_uid",
            Collections.emptyList(),
            Long::valueOf,
            new ArrayList<>());
        terminalsInfos = server.terminalsInfos();
        limit = session.params().get(
            "limit",
            20,
            NonNegativeIntegerValidator.INSTANCE);
        jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        Set<Long> serviceIds = session.params().getAll(
            "subservice_ids",
            Collections.emptySet(),
            Long::valueOf,
            new HashSet<>());
        if (serviceIds.isEmpty()) {
            this.serviceIds = terminalsInfos.servicesWhitelist();
        } else {
            serviceIds.retainAll(terminalsInfos.servicesWhitelist());
            this.serviceIds = serviceIds;
        }
        showYandexPay = session.params().getBoolean("show-yandexpay", false);
        afterTimestamp = session.params().get(
            "after-timestamp",
            0L,
            PositiveLongValidator.INSTANCE);
        beforeTimestamp = session.params().get(
            "before-timestamp",
            0L,
            PositiveLongValidator.INSTANCE);
        startPosTimestamp = session.params().getLong("order_id_keyset", null);
        if (startPosTimestamp == null) {
            startPosPurchaseToken = null;
        } else {
            startPosPurchaseToken = session.params().get(
                "created_keyset",
                NonEmptyValidator.INSTANCE);
        }
    }

    public OhioBackend server() {
        return server;
    }

    public ProxySession session() {
        return session;
    }

    public TerminalsInfos terminalsInfos() {
        return terminalsInfos;
    }

    public long uid() {
        return uid;
    }

    public boolean hideFamilyPayments() {
        return hideFamilyPayments;
    }

    public List<Long> initiatorsUids() {
        return initiatorsUids;
    }

    public int limit() {
        return limit;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public Set<Long> serviceIds() {
        return serviceIds;
    }

    public boolean showYandexPay() {
        return showYandexPay;
    }

    public long afterTimestamp() {
        return afterTimestamp;
    }

    public long beforeTimestamp() {
        return beforeTimestamp;
    }

    public void addStartPosToQuery(final QueryConstructor query)
        throws BadRequestException
    {
        query.append("dp", "multi(timestamp,purchase_token pos)");
        query.append("sort", "pos");
        if (startPosTimestamp != null) {
            query.append(
                "dp",
                "const(" + startPosTimestamp + " start_timestamp_str)");
            query.append(
                "dp",
                "to-long(start_timestamp_str start_timestamp)");
            query.append(
                "dp",
                "const(" + startPosPurchaseToken + " start_purchase_token)");
            query.append(
                "dp",
                "multi(start_timestamp,start_purchase_token start_pos)");
            query.append("postfilter", "pos lessthan start_pos");
        }
    }

    public String baseQuery() {
        return null;
    }

    public boolean postfilterServices() {
        int totalServices = terminalsInfos.servicesWhitelist().size();
        return !showYandexPay && serviceIds.size() > (totalServices / 2);
    }
}

