package ru.yandex.ohio.backend;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.FakeAsyncConsumer;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonDouble;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.StringMapValuesStorage;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public abstract class ListingHandlerBase
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final DateTimeFormatter DATE_TIME_FORMATTER =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZZ").withZoneUTC();
    private static final DateTimeFormatter UPDATED_TIME_FORMATTER =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZZ").withZoneUTC();
    private static final long YANDEX_ACCOUNT_TOPUP = 1535;
    private static final long YANDEX_ACCOUNT_WITHDRAW = 1534;
    private static final long PROMO_PAYMENT_METHOD_ID = 1526;
    private static final Set<String> ACCEPTABLE_STATUSES =
        new HashSet<>(Arrays.asList("paid", "cancelled", "refunded", "hold"));
    private static final Map<String, String> YANDEX_PAY_STATUS_MAPPING = Map.of(
        "captured", "hold",
        "charged", "paid",
        "cancelled", "cancelled");
    private static final ItemAlias[] ALIASES = ItemAlias.values();
    public static final JsonObject EMPTY_RESULT;

    static {
        try {
            EMPTY_RESULT = TypesafeValueContentHandler.parse(
                "{\"orders\":[],\"next\":{}}");
        } catch (JsonException e) {
            throw new RuntimeException(e);
        }
    }

    protected final OhioBackend server;

    protected ListingHandlerBase(final OhioBackend server) {
        this.server = server;
    }

    @Override
    public FakeAsyncConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new FakeAsyncConsumer<>(request);
    }

    public static int increaseLimit(final int limit) {
        return increaseLimit(limit, 2);
    }

    public static int increaseLimit(final int limit, final int positions) {
        int newLimit = limit << positions;
        if (newLimit < 0) {
            newLimit = Integer.MAX_VALUE;
        }
        return newLimit;
    }

    protected void handle(final ListingContext listingContext)
        throws HttpException
    {
        if (listingContext.serviceIds().isEmpty()
            || listingContext.limit() == 0)
        {
            new ResultPrinter(
                listingContext.session(),
                listingContext.jsonType())
                .completed(EMPTY_RESULT);
        } else {
            listing(listingContext, increaseLimit(listingContext.limit()));
        }
    }

    private void listing(final ListingContext listingContext, final int limit)
        throws HttpException
    {
        ProxySession session = listingContext.session();
        session.logger().info(
            "Trying to list " + limit + " documents with "
            + listingContext.limit() + " documents required");
        boolean postfilterServices = listingContext.postfilterServices();
        StringBuilder baseFilter =
            new StringBuilder("((has_payment_timestamp:1");
        if (!postfilterServices) {
            baseFilter.append(" AND service_id:(");
            boolean empty = true;
            for (Long serviceId: listingContext.serviceIds()) {
                if (empty) {
                    empty = false;
                } else {
                    baseFilter.append(' ');
                }
                baseFilter.append(serviceId.longValue());
            }
            baseFilter.append(')');
        }
        baseFilter.append(')');
        if (listingContext.showYandexPay()
            && listingContext.serviceIds.contains(1042L))
        {
            baseFilter.append(
                " OR (source:yandexpay AND "
                + "last_payment_status:(charged OR cancelled OR captured))");
        }
        baseFilter.append(')');
        String baseQuery = listingContext.baseQuery();
        if (baseQuery != null) {
            baseFilter.append(" AND (");
            baseFilter.append(baseQuery);
            baseFilter.append(')');
        }
        if (listingContext.hideFamilyPayments()) {
            baseFilter.append(" AND NOT has_initiator_uid:1");
        }
        List<Long> initiatorsUids = listingContext.initiatorsUids();
        int size = initiatorsUids.size();
        if (size > 0) {
            baseFilter.append(" AND initiator_uid:(");
            for (int i = 0; i < size; ++i) {
                if (i > 0) {
                    baseFilter.append(" OR ");
                }
                baseFilter.append(initiatorsUids.get(i).longValue());
            }
            baseFilter.append(')');
        }
        String queryText = new String(baseFilter);
        long uid = listingContext.uid();
        QueryConstructor query =
            new QueryConstructor(
                "/search?json-type=dollar&service=ohio_index&get=*&prefix="
                + uid + "&length=" + limit);
        listingContext.addStartPosToQuery(query);
        query.append("text", queryText);
        if (postfilterServices) {
            StringBuilder sb = new StringBuilder("service_id in ");
            boolean empty = true;
            for (Long serviceId: listingContext.serviceIds()) {
                if (empty) {
                    empty = false;
                } else {
                    sb.append(',');
                }
                sb.append(serviceId.longValue());
            }
            query.append("postfilter", new String(sb));
        }
        long afterTimestamp = listingContext.afterTimestamp();
        if (afterTimestamp != 0L) {
            query.append(
                "postfilter",
                "timestamp >= " + afterTimestamp * 1000L);
        }
        long beforeTimestamp = listingContext.beforeTimestamp();
        if (beforeTimestamp != 0L) {
            query.append(
                "postfilter",
                "timestamp <= " + beforeTimestamp * 1000L);
        }
        AsyncClient client = server.searchClient().adjust(session.context());
        server.parallelRequest(
            session,
            new PlainUniversalSearchProxyRequestContext(
                new User("ohio_index", new LongPrefix(uid)),
                null,
                true,
                client,
                session.logger()),
            new BasicAsyncRequestProducerGenerator(query.toString()),
            SearchResultConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new Callback(listingContext, limit));
    }

    private static JsonObject fixCurrency(final String currency) {
        if (currency == null) {
            return JsonNull.INSTANCE;
        } else if ("RUR".equals(currency)) {
            return new JsonString("RUB");
        } else {
            return new JsonString(currency);
        }
    }

    private static String guessPaymentMethod(
        final String trustPaymentId,
        final Set<String> trustGroupIds,
        final TerminalInfo terminalInfo,
        // terminalInfo.serviceId() may be -1
        final long serviceId)
    {
        if (trustGroupIds.contains(trustPaymentId)) {
            return "composite";
        }

        if (terminalInfo.paymentMethodId() == YANDEX_ACCOUNT_TOPUP
            || terminalInfo.paymentMethodId() == YANDEX_ACCOUNT_WITHDRAW)
        {
            return "yandex_account";
        }

        if (terminalInfo.paymentMethodId() == PROMO_PAYMENT_METHOD_ID) {
            switch ((int) serviceId) {
                // Zapravki
                case 621:
                case 631:
                case 636:
                case 637:
                // Afisha
                case 118:
                case 126:
                case 131:
                case 617:
                case 638:
                case 712:
                case 717:
                    return "virtual::new_promocode";
                // Kinopoisk
                case 136:
                case 600:
                case 713:
                case 735:
                    if (terminalInfo.terminalId() == 57000021L) {
                        return "virtual::kinopoisk_subs_discounts";
                    } else {
                        return "virtual::kinopoisk_card_discounts";
                    }
                case 1125:
                    return "wrapper::bnpl";
                default:
                    break;
            }
        }

        return terminalInfo.paymentMethodName();
    }

    public static ItemAlias detectAlias(
        final String name,
        final long serviceId,
        final JsonMap orderInfo)
        throws JsonException
    {
        for (ItemAlias alias: ALIASES) {
            if (alias.testItem(name, serviceId, orderInfo)) {
                return alias;
            }
        }
        return null;
    }

    public static void foldExtendedSubscription(
        final JsonList items,
        final int extendedSubscriptionIdx,
        final int transportingIdx)
        throws JsonException
    {
        if (extendedSubscriptionIdx != -1
            && transportingIdx != -1
            && items.get(extendedSubscriptionIdx).get("currency").equals(
                items.get(transportingIdx).get("currency")))
        {
            JsonMap transporting = items.get(transportingIdx).asMap();
            transporting.put(
                "amount",
                new JsonString(
                    Double.toString(
                        transporting.getDouble("amount")
                        + items.get(extendedSubscriptionIdx).get("amount")
                            .asDouble())));
            transporting.put(
                "price",
                new JsonString(
                    Double.toString(
                        transporting.getDouble("price")
                        + items.get(extendedSubscriptionIdx).get("price")
                            .asDouble())));
            items.remove(extendedSubscriptionIdx);
        } else if (extendedSubscriptionIdx != -1) {
            items.get(extendedSubscriptionIdx)
                .asMap()
                .put("hidden", JsonBoolean.TRUE);
        }
    }

    private class Callback extends AbstractProxySessionCallback<SearchResult> {
        private final ListingContext listingContext;
        private final int limit;
        private final PrefixedLogger logger;
        private long lastTimestamp;
        private String lastPurchaseToken;

        Callback(final ListingContext listingContext, final int limit) {
            super(listingContext.session());
            this.listingContext = listingContext;
            this.limit = limit;
            logger = session.logger();
        }

        private JsonList convertItems(
            final JsonList rows,
            final JsonObject currency,
            final Map<Long, JsonMap> productsMap,
            final long serviceId,
            final boolean putImagePath,
            final PricePostprocessor pricePostprocessor)
            throws JsonException
        {
            int size = rows.size();
            if (size == 0) {
                return JsonList.EMPTY;
            }
            int transportingIdx = -1;
            int extendedSubscriptionIdx = -1;
            JsonList items =
                new JsonList(BasicContainerFactory.INSTANCE, size);
            for (int i = 0; i < size; ++i) {
                JsonMap row = rows.get(i).asMap();
                String nds = row.getString("fiscal_nds", "");
                String name = row.getString("fiscal_title", "");
                if ("none".equals(name)) {
                    name = "";
                }
                JsonMap order =
                    Objects.requireNonNullElse(
                        row.get("order").asMapOrNull(),
                        JsonMap.EMPTY);
                Long serviceProductId =
                    order.getLong("service_product_id", null);
                JsonMap product = productsMap.get(serviceProductId);
                String productNds = "nds_none";
                if (product != null && name.isEmpty()) {
                    productNds = product.getString("fiscal_nds", "");
                    name = product.getString("fiscal_title", "");
                    if (name.isEmpty()) {
                        name = product.getString("name", "");
                    }
                }
                if (nds.isEmpty()) {
                    nds = productNds;
                }
                JsonMap item = new JsonMap(BasicContainerFactory.INSTANCE);
                double amount =
                    pricePostprocessor.roundPrice(row.getDouble("amount", 0d));
                item.put(
                    "amount",
                    new JsonString(Double.toString(amount)));
                double price =
                    pricePostprocessor.roundPrice(row.getDouble("price", 0d));
                item.put(
                    "price",
                    new JsonString(Double.toString(price)));
                item.put("currency", currency);
                item.put("nds", new JsonString(nds));

                ItemAlias alias = detectAlias(name, serviceId, order);
                String nameReplacement;
                if (alias == null) {
                    nameReplacement = null;
                } else {
                    nameReplacement = alias.nameReplacement();
                }
                if (nameReplacement != null) {
                    name = nameReplacement;
                } else if (name.isEmpty()) {
                    name = "none";
                }
                item.put("name", new JsonString(name));
                if (alias == ItemAlias.TRANSPORTING_FEE) {
                    transportingIdx = i;
                } else if (alias == ItemAlias.EXTENDED_SUBSCRIPTION) {
                    extendedSubscriptionIdx = i;
                }

                if (alias != null && alias.export()) {
                    item.put("alias", new JsonString(alias.aliasName()));
                }

                if (putImagePath) {
                    item.put("image_path", JsonNull.INSTANCE);
                }
                items.add(item);
            }
            foldExtendedSubscription(
                items,
                extendedSubscriptionIdx,
                transportingIdx);
            return items;
        }

        // Returns null if doc must be skipped
        // TODO: errors stater
        private JsonMap convertYandexPayDoc(
            final StringMapValuesStorage mapDoc)
            throws JsonException, ParseException
        {
            JsonMap doc = new JsonMap(BasicContainerFactory.INSTANCE);
            doc.put("source", new JsonString("yandexpay"));
            doc.put(
                "order_revision",
                new JsonLong(mapDoc.getLong("order_revision", 1L)));
            doc.put(
                "order_id",
                new JsonLong(mapDoc.getLong("sequence_number")));

            String purchaseToken = mapDoc.getString("purchase_token");
            doc.put("trust_purchase_token", new JsonString(purchaseToken));
            doc.put("trust_payment_id", new JsonString(purchaseToken));
            doc.put("subservice_id", new JsonString("1042"));

            String merchantId = mapDoc.getString("merchant_id", null);
            if (merchantId == null) {
                logger.warning("No merchant_id found for " + mapDoc);
                return null;
            }
            doc.put("merchant_id", new JsonString(merchantId));

            String gatewayName = mapDoc.getString("gateway_name", null);
            if (gatewayName != null) {
                doc.put("gateway_name", new JsonString(gatewayName));
            }

            double amount = mapDoc.getDouble("amount", 0d);
            if (amount <= 0d) {
                logger.warning("Zero amount for " + mapDoc);
                return null;
            }
            doc.put("total", new JsonString(Double.toString(amount)));
            JsonString currency = new JsonString(mapDoc.getString("currency"));
            doc.put("currency", currency);

            long timestamp = mapDoc.getLong("timestamp");
            doc.put(
                "created",
                new JsonString(DATE_TIME_FORMATTER.print(timestamp)));
            doc.put(
                "updated",
                new JsonString(
                    UPDATED_TIME_FORMATTER.print(
                        mapDoc.getLong("update_timestamp", timestamp))));

            String status = mapDoc.getString("last_payment_status", null);
            String remappedStatus = YANDEX_PAY_STATUS_MAPPING.get(status);
            if (remappedStatus == null) {
                logger.warning(
                    "Unexpected status " + status
                    + ", skipping doc " + mapDoc);
                return null;
            }
            doc.put("status", new JsonString(remappedStatus));

            JsonList yandexPayItems =
                Objects.requireNonNullElse(
                    mapDoc.get(
                        "yandexpay_items",
                        JsonNull.INSTANCE,
                        TypesafeValueContentHandler::parse)
                        .asListOrNull(),
                    JsonList.EMPTY);
            int size = yandexPayItems.size();
            JsonList items;
            if (size == 0) {
                JsonMap item = new JsonMap(BasicContainerFactory.INSTANCE);
                item.put("name", new JsonString("none"));
                item.put("image_path", JsonNull.INSTANCE);
                // XXX
                item.put("currency", currency);
                item.put("nds", JsonString.EMPTY);
                item.put("price", new JsonString(Double.toString(amount)));
                item.put("amount", new JsonString(Double.toString(amount)));
                items = new JsonList(BasicContainerFactory.INSTANCE, 1);
                items.add(item);
            } else {
                items = new JsonList(BasicContainerFactory.INSTANCE, size);
                for (int i = 0; i < size; ++i) {
                    JsonMap yandexPayItem = yandexPayItems.get(i).asMap();
                    String label = yandexPayItem.getString("label", "");
                    if (label.isEmpty()) {
                        label = "none";
                    }
                    JsonMap item = new JsonMap(BasicContainerFactory.INSTANCE);
                    item.put("name", new JsonString(label));
                    double itemAmount = yandexPayItem.getDouble("amount");
                    item.put("image_path", JsonNull.INSTANCE);
                    // XXX
                    item.put("currency", currency);
                    // XXX
                    item.put("nds", JsonString.EMPTY);
                    item.put(
                        "price",
                        new JsonString(Double.toString(itemAmount)));
                    item.put(
                        "amount",
                        new JsonString(Double.toString(itemAmount)));
                    items.add(item);
                }
            }
            doc.put("items", items);

            JsonMap serviceData = new JsonMap(BasicContainerFactory.INSTANCE);
            serviceData.put("user_account", JsonString.EMPTY);
            // XXX
            serviceData.put("payment_method", new JsonString("unknown"));
            serviceData.put("trust_group_id", JsonString.EMPTY);
            serviceData.put("cashback_amount", JsonLong.ZERO);
            serviceData.put("trust_payment_id", new JsonString(purchaseToken));
            serviceData.put("composite_payment_id", JsonLong.ZERO);
            doc.put("service_data", serviceData);
            doc.put("customer_uid", new JsonLong(listingContext.uid()));
            doc.put("service_revision", JsonLong.ONE);
            doc.put("merchant_uid", JsonLong.ONE);
            doc.put("description", JsonString.EMPTY);
            doc.put("refunds", JsonList.EMPTY);

            logger.info(
                "Yandex.Pay doc converted, message_id: " + purchaseToken);
            lastTimestamp = timestamp;
            lastPurchaseToken = purchaseToken;
            return doc;
        }

        // Returns null if doc must be skipped
        // TODO: errors stater
        private JsonMap convertDoc(
            final StringMapValuesStorage mapDoc,
            final Set<String> trustGroupIds)
            throws JsonException, ParseException
        {
            String source = mapDoc.getString("source", null);
            if ("yandexpay".equals(source)) {
                return convertYandexPayDoc(mapDoc);
            }
            long terminalId = mapDoc.getLong("terminal_id");
            long serviceId = mapDoc.getLong("service_id");
            String purchaseToken = mapDoc.getString("purchase_token");

            // We don't check that service is known, because our search request
            // contained only known services
            TerminalInfo terminalInfo =
                listingContext.terminalsInfos().getTerminalInfo(
                    terminalId,
                    serviceId);
            if (terminalInfo == null) {
                logger.warning(
                    "Skipping doc " + mapDoc
                    + ": terminal info was not found for transaction");
                return null;
            }

            JsonList rows =
                Objects.requireNonNullElse(
                    mapDoc.get(
                        "rows",
                        JsonNull.INSTANCE,
                        TypesafeValueContentHandler::parse)
                        .asListOrNull(),
                    JsonList.EMPTY);
            int rowsSize = rows.size();

            String dataFormatVersion = mapDoc.getString("data_format_version");
            boolean importedRefund = dataFormatVersion.equals("3");
            JsonList refundsList;
            if (importedRefund) {
                refundsList = JsonList.EMPTY;
            } else {
                refundsList =
                    Objects.requireNonNullElse(
                        mapDoc.get(
                            "refunds",
                            JsonNull.INSTANCE,
                            TypesafeValueContentHandler::parse)
                            .asListOrNull(),
                        JsonList.EMPTY);
            }
            int refundsSize = refundsList.size();

            if (rowsSize == 0) {
                boolean empty = true;
                for (int i = 0; i < refundsSize && empty; ++i) {
                    JsonList refundRows =
                        Objects.requireNonNullElse(
                            refundsList.get(i).get("rows").asListOrNull(),
                            JsonList.EMPTY);
                    empty = refundRows.isEmpty();
                }
                if (empty) {
                    logger.warning(
                        "Skipping doc " + mapDoc + ": no rows found");
                    return null;
                }
            }

            long paymentMethodId = terminalInfo.paymentMethodId();
            String trustPaymentId = mapDoc.getString("trust_payment_id");
            String paymentMethod =
                mapDoc.getString("normalized_payment_method", null);
            if (paymentMethod == null) {
                logger.warning(
                    "No normalized_payment_method found for"
                    + " purchase_token " + purchaseToken);
                paymentMethod = guessPaymentMethod(
                    trustPaymentId,
                    trustGroupIds,
                    terminalInfo,
                    serviceId);
                logger.info("Guessed payment method: " + paymentMethod);
            }
            switch (paymentMethod) {
                case "virtual::new_promocode":
                    if (serviceId != 621L
                        && serviceId != 631L
                        && serviceId != 636L
                        && serviceId != 637L)
                    {
                        logger.warning(
                            "Skipping doc " + mapDoc + ": new_promocode");
                        return null;
                    }
                    break;
                case "yandex_account":
                    if (paymentMethodId == YANDEX_ACCOUNT_TOPUP) {
                        paymentMethod = "yandex_account_topup";
                    } else if (paymentMethodId == YANDEX_ACCOUNT_WITHDRAW) {
                        paymentMethod = "yandex_account_withdraw";
                    } else {
                        logger.warning(
                            "Skipping doc " + mapDoc
                            + ": unknown yandex_account payment method "
                            + paymentMethodId);
                        return null;
                    }
                    break;
                case "google_pay_token":
                    paymentMethod = "google_pay";
                    break;
                case "apple_token":
                    paymentMethod = "apple_pay";
                    break;
                default:
                    if (server.paymentMethodsBlacklist().contains(
                        paymentMethod))
                    {
                        logger.warning(
                            "Skipping doc " + mapDoc + ": payment method "
                            + paymentMethod + " blacklisted");
                        return null;
                    }
                    break;
            }

            // Skip Yandex Bank transactions, keep Plus transactions
            if (serviceId == 1129
                && !paymentMethod.equals("yandex_account_topup")
                && !paymentMethod.equals("yandex_account_withdraw"))
            {
                logger.warning(
                    "Skipping doc " + mapDoc + ": payment method "
                    + paymentMethod + " not allowed for Yandex Bank");
                return null;
            }

            double amount = mapDoc.getDouble("postauth_amount", 0d);
            Long cancelTimestamp =
                mapDoc.getLong("cancel_timestamp", null);
            Long paymentTimestamp =
                mapDoc.getLong("payment_timestamp", null);
            Long postauthTimestamp =
                mapDoc.getLong("postauth_timestamp", null);
            final String status;
            if (cancelTimestamp == null) {
                if (refundsSize > 0) {
                    status = "refunded";
                } else if (postauthTimestamp == null) {
                    if (paymentTimestamp == null) {
                        status = "initial";
                    } else {
                        status = "hold"; // a.k.a. "authorized"
                    }
                } else {
                    status = "paid"; // a.k.a. "postauthorized"
                }
            } else {
                if (postauthTimestamp == null) {
                    status = "not authorized";
                } else if (importedRefund) {
                    status = "paid"; // can't import refunds properly, skip it
                } else {
                    if (amount > 0d) {
                        status = "refunded";
                    } else {
                        status = "cancelled";
                    }
                }
            }
            if (!ACCEPTABLE_STATUSES.contains(status)) {
                logger.warning(
                    "Skipping doc " + mapDoc
                    + ": payment status \"" + status + "\" is not acceptable");
                return null;
            }

            if (amount == 0d) {
                // This is refund, take value from amount
                amount = mapDoc.getDouble("amount");
            }

            String originalCurrency = mapDoc.getString("currency");
            PricePostprocessor pricePostprocessor;
            if (paymentMethod.equals("yandex_account_topup")
                || paymentMethod.equals("yandex_account_withdraw"))
            {
                pricePostprocessor =
                    new YandexPlusPricePostprocessor(originalCurrency);
            } else {
                pricePostprocessor = IdentityPricePostprocessor.INSTANCE;
            }
            amount = pricePostprocessor.roundPrice(amount);
            if (amount <= 0d && pricePostprocessor.skipZeroPrice()) {
                logger.warning(
                    "Skipping doc " + mapDoc + ": skipping zero amount");
                return null;
            }

            JsonMap doc = new JsonMap(BasicContainerFactory.INSTANCE);
            doc.put("source", new JsonString("trust"));
            doc.put("subservice_id", new JsonString(Long.toString(serviceId)));
            long sequenceNumber = mapDoc.getLong("sequence_number");
            if (sequenceNumber < Integer.MAX_VALUE * 1000L) {
                // This is historical data, no precise timing available, let's
                // randomize it with purchase_token hash code
                sequenceNumber *= 1000L;
                sequenceNumber += purchaseToken.hashCode() & 0xfffffL;
            }
            doc.put("order_id", new JsonLong(sequenceNumber));
            doc.put("trust_purchase_token", new JsonString(purchaseToken));
            doc.put("merchant_uid", JsonLong.ONE);
            JsonLong uid = new JsonLong(listingContext.uid());
            doc.put("service_merchant_id", uid);
            doc.put("customer_uid", uid);
            doc.put("description", JsonString.EMPTY);
            doc.put("payments_order_id", JsonLong.ONE);
            doc.put("status", new JsonString(status));
            doc.put(
                "order_revision",
                new JsonLong(mapDoc.getLong("order_revision", 1L)));
            doc.put("service_revision", JsonLong.ONE);
            long timestamp = mapDoc.getLong("timestamp");
            doc.put(
                "created",
                new JsonString(DATE_TIME_FORMATTER.print(timestamp)));
            doc.put(
                "updated",
                new JsonString(
                    UPDATED_TIME_FORMATTER.print(
                        mapDoc.getLong("update_timestamp", timestamp))));
            doc.put("trust_payment_id", new JsonString(trustPaymentId));

            Long sponsorUid = mapDoc.getLong("sponsor_uid", null);
            if (sponsorUid != null) {
                doc.put("sponsor_uid", new JsonLong(sponsorUid.longValue()));
            }
            Long initiatorUid = mapDoc.getLong("initiator_uid", null);
            if (initiatorUid != null) {
                doc.put(
                    "initiator_uid",
                    new JsonLong(initiatorUid.longValue()));
            }

            JsonMap serviceData = new JsonMap(BasicContainerFactory.INSTANCE);
            serviceData.put(
                "user_account",
                new JsonString(mapDoc.getString("user_account", "")));
            serviceData.put(
                "cashback_amount",
                new JsonDouble(
                    pricePostprocessor.roundPrice(
                        mapDoc.getDouble("cashback_amount", 0d))));
            serviceData.put(
                "trust_group_id",
                new JsonString(mapDoc.getString("trust_group_id", "")));
            serviceData.put("payment_method", new JsonString(paymentMethod));
            serviceData.put(
                "composite_payment_id",
                new JsonLong(mapDoc.getLong("composite_payment_id", 0L)));
            serviceData.put(
                "trust_payment_id",
                new JsonString(trustPaymentId));

            doc.put("service_data", serviceData);

            doc.put("total", new JsonString(Double.toString(amount)));
            JsonObject currency = fixCurrency(originalCurrency);
            doc.put("currency", currency);

            JsonList products =
                Objects.requireNonNullElse(
                    mapDoc.get(
                        "products",
                        JsonNull.INSTANCE,
                        TypesafeValueContentHandler::parse)
                        .asListOrNull(),
                    JsonList.EMPTY);
            int productsSize = products.size();
            Map<Long, JsonMap> productsMap;
            if (productsSize == 0) {
                productsMap = Collections.emptyMap();
            } else {
                productsMap = new HashMap<>(productsSize << 1);
                for (int i = 0; i < productsSize; ++i) {
                    JsonMap product = products.get(i).asMap();
                    Long productId = product.getLong("id", null);
                    if (productId != null) {
                        productsMap.put(productId, product);
                    }
                }
            }

            doc.put(
                "items",
                convertItems(
                    rows,
                    currency,
                    productsMap,
                    serviceId,
                    true,
                    pricePostprocessor));

            JsonList refunds;
            if (refundsSize == 0) {
                refunds = JsonList.EMPTY;
            } else {
                refunds =
                    new JsonList(BasicContainerFactory.INSTANCE, refundsSize);
                for (int i = 0; i < refundsSize; ++i) {
                    JsonMap refund = refundsList.get(i).asMap();
                    JsonMap refundItem =
                        new JsonMap(BasicContainerFactory.INSTANCE);
                    refundItem.put(
                        "items",
                        convertItems(
                            Objects.requireNonNullElse(
                                refund.get("rows").asListOrNull(),
                                JsonList.EMPTY),
                            currency,
                            productsMap,
                            serviceId,
                            false,
                            pricePostprocessor));
                    refundItem.put(
                        "trust_refund_id",
                        refund.get("trust_refund_id"));
                    refundItem.put(
                        "refund_status",
                        new JsonString("completed"));
                    refundItem.put(
                        "total",
                        new JsonString(
                            Double.toString(
                                pricePostprocessor.roundPrice(
                                    refund.getDouble("amount", 0d)))));
                    refundItem.put(
                        "currency",
                        fixCurrency(refund.get("currency").asStringOrNull()));
                    refunds.add(refundItem);
                }
            }
            doc.put("refunds", refunds);
            logger.info("Doc converted, purchase_token: " + purchaseToken);
            lastTimestamp = timestamp;
            lastPurchaseToken = purchaseToken;
            return doc;
        }

        private JsonList filterDocuments(
            final List<SearchDocument> docs)
        {
            int size = docs.size();
            Set<String> trustGroupIds = new HashSet<>(size);
            for (int i = 0; i < size; ++i) {
                String trustGroupId =
                    docs.get(i).attrs().get("trust_group_id");
                if (trustGroupId != null && !trustGroupId.isEmpty()) {
                    trustGroupIds.add(trustGroupId);
                }
            }
            int limit = listingContext.limit();
            JsonList result =
                new JsonList(BasicContainerFactory.INSTANCE, size);
            for (int i = 0; i < size; ++i) {
                // TODO: errors stater
                Map<String, String> doc = docs.get(i).attrs();
                try {
                    JsonMap jsonDoc = convertDoc(
                        new StringMapValuesStorage(doc),
                        trustGroupIds);
                    if (jsonDoc != null) {
                        result.add(jsonDoc);
                        if (result.size() >= limit) {
                            break;
                        }
                    }
                } catch (JsonException | ParseException e) {
                    listingContext.session().logger().log(
                        Level.WARNING,
                        "Skipping doc " + doc,
                        e);
                }
            }
            return result;
        }

        @Override
        public void completed(final SearchResult result) {
            List<SearchDocument> hits = result.hitsArray();
            JsonList filtered = filterDocuments(hits);
            int size = filtered.size();
            listingContext.session().logger().info(
                "Got " + hits.size() + " documents, after filtration "
                + size + " documents left");
            if (size < listingContext.limit()
                && hits.size() >= limit)
            {
                // Not enough data, increase limit and repeat
                // TODO: Retries stater, count and max
                try {
                    listing(listingContext, increaseLimit(limit));
                } catch (HttpException e) {
                    failed(e);
                }
                return;
            }

            if (size == 0) {
                new ResultPrinter(
                    listingContext.session(),
                    listingContext.jsonType())
                    .completed(EMPTY_RESULT);
                return;
            }

            JsonMap next = new JsonMap(BasicContainerFactory.INSTANCE, 3);
            next.put("order_id_keyset", new JsonLong(lastTimestamp));
            next.put("created_keyset", new JsonString(lastPurchaseToken));

            JsonMap data = new JsonMap(BasicContainerFactory.INSTANCE, 3);
            data.put("orders", filtered);
            data.put("next", next);

            new ResultPrinter(
                listingContext.session(),
                listingContext.jsonType())
                .completed(data);
        }
    }
}

