package ru.yandex.ohio.backend;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.util.HashSet;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.charset.Decoder;
import ru.yandex.collection.Pattern;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.http.util.server.ExternalDataSubscriber;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.ohio.backend.config.ImmutableOhioBackendConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;

public class OhioBackend
    extends UniversalSearchProxy<ImmutableOhioBackendConfig>
    implements ExternalDataSubscriber
{
    private final HttpHost producerStoreHost;
    private final AsyncClient producerStoreClient;
    private final HttpHost darkspiritHost;
    private final AsyncClient darkspiritClient;
    private final String checkUrlBase;
    private final HttpHost fiscalStoragesHost;
    private final AsyncClient fiscalStoragesClient;
    private final HttpHost dyngoHost;
    private final AsyncClient dyngoClient;
    private final HttpHost fnslinkerHost;
    private final AsyncClient fnslinkerClient;
    private volatile Set<Long> servicesWhitelist;
    private volatile Set<String> paymentMethodsBlacklist;
    private volatile TerminalsInfos terminalsInfos;

    public OhioBackend(final ImmutableOhioBackendConfig config)
        throws ConfigException, HttpException, IOException, JsonException
    {
        super(config);

        ImmutableHttpHostConfig producerStoreConfig =
            config.producerStoreConfig();
        producerStoreHost = producerStoreConfig.host();
        producerStoreClient = client("ProducerStore", producerStoreConfig);

        ImmutableHttpHostConfig darkspiritConfig = config.darkspiritConfig();
        darkspiritHost = darkspiritConfig.host();
        darkspiritClient = client("Darkspirit", darkspiritConfig);
        checkUrlBase = config.checkUrlBaseString();

        ImmutableHttpHostConfig fiscalStoragesConfig =
            config.fiscalStoragesConfig();
        fiscalStoragesHost = fiscalStoragesConfig.host();
        fiscalStoragesClient = client("FiscalStorages", fiscalStoragesConfig);

        ImmutableHttpHostConfig dyngoConfig = config.dyngoConfig();
        dyngoHost = dyngoConfig.host();
        dyngoClient = client("Dyngo", dyngoConfig);

        ImmutableHttpHostConfig fnslinkerConfig = config.fnslinkerConfig();
        fnslinkerHost = fnslinkerConfig.host();
        fnslinkerClient = client("FnsLinker", fnslinkerConfig);

        subscribeForExternalDataUpdates("bunker", this);

        try (BufferedReader reader =
                Files.newBufferedReader(config.terminalsInfosPath()))
        {
            updateTerminalsInfos(reader);
        }

        register(
            new Pattern<>("/dump-user-data", false),
            new DumpUserDataHandler(this));

        register(
            new Pattern<>("/update-terminals-infos", false),
            new UpdateTerminalsInfosHandler(this));

        register(
            new Pattern<>("/import-refunds", false),
            new ImportRefundsHandler(this));

        register(
            new Pattern<>("/services", false),
            new DirectServicesHandler(this));

        register(
            new Pattern<>("/familypay_users", false),
            new DirectFamilypayUsersHandler(this));

        register(
            new Pattern<>("/orders", false),
            new DirectOrdersHandler(this));

        register(
            new Pattern<>("/fns_orders", false),
            new DirectFnsOrdersHandler(this));

        register(
            new Pattern<>("/yandex_account_orders", false),
            new DirectYandexAccountOrdersHandler(this));

        register(
            new Pattern<>("/cashback_balance", false),
            new DirectCashbackBalanceHandler(this));

        register(
            new Pattern<>("/payment", false),
            new DirectPaymentHandler(this));

        register(
            new Pattern<>("/fns_binding_status", false),
            new DirectFnsBindingStatusHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*{path_regex:/v1/customer/[0-9]+/services}"),
            new ServicesHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*"
                + "{path_regex:/v1/customer/[0-9]+/familypay_users}"),
            new FamilypayUsersHandler(this));

        register(
            new Pattern<>("/receipts", false),
            new ReceiptsHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*{path_regex:/v1/customer/[0-9]+/orders}"),
            new OrdersHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*{path_regex:/v1/customer/[0-9]+/fns_orders}"),
            new FnsOrdersHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*"
                + "{path_regex:/v1/customer/[0-9]+/yandex_account_orders}"),
            new YandexAccountOrdersHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*"
                + "{path_regex:/v1/customer/[0-9]+/cashback_balance}"),
            new CashbackBalanceHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*"
                + "{path_regex:/v1/customer/[0-9]+/payment}"),
            new PaymentHandler(this));

        register(
            RequestPatternParser.INSTANCE.apply(
                "/v1/customer/*"
                + "{path_regex:/v1/customer/[0-9]+/fns_binding_status}"),
            new FnsBindingStatusHandler(this));
    }

    @Override
    public void updateExternalData(
        final ByteArrayProcessableWithContentType data)
        throws HttpException, IOException
    {
        Decoder decoder = new Decoder(data.contentType().getCharset());
        data.data().processWith(decoder);
        try {
            JsonMap map = TypesafeValueContentHandler.parse(decoder).asMap();

            JsonList whitelist = map.get("servicesWhitelist").asList();
            int size = whitelist.size();
            Set<Long> servicesWhitelist = new HashSet<>(size << 1);
            for (int i = 0; i < size; ++i) {
                servicesWhitelist.add(whitelist.get(i).asLong());
            }

            JsonList blacklist = map.get("paymentsBlacklist").asList();
            size = blacklist.size();
            Set<String> paymentMethodsBlacklist = new HashSet<>(size << 1);
            for (int i = 0; i < size; ++i) {
                paymentMethodsBlacklist.add(blacklist.get(i).asString());
            }

            TerminalsInfos terminalsInfos;
            try (BufferedReader reader =
                    Files.newBufferedReader(config.terminalsInfosPath()))
            {
                terminalsInfos = new TerminalsInfos(servicesWhitelist, reader);
            }

            this.servicesWhitelist = servicesWhitelist;
            this.paymentMethodsBlacklist = paymentMethodsBlacklist;
            this.terminalsInfos = terminalsInfos;
        } catch (JsonException e) {
            throw new ServiceUnavailableException(
                "Failed to parse json < " + decoder + '>',
                e);
        }
    }

    public Set<String> paymentMethodsBlacklist() {
        return paymentMethodsBlacklist;
    }

    public TerminalsInfos terminalsInfos() {
        return terminalsInfos;
    }

    public void updateTerminalsInfos(final BufferedReader reader)
        throws IOException, JsonException
    {
        TerminalsInfos terminalsInfos =
            new TerminalsInfos(servicesWhitelist, reader);
        this.terminalsInfos = terminalsInfos;
    }

    public HttpHost producerStoreHost() {
        return producerStoreHost;
    }

    public AsyncClient producerStoreClient() {
        return producerStoreClient;
    }

    public HttpHost darkspiritHost() {
        return darkspiritHost;
    }

    public AsyncClient darkspiritClient() {
        return darkspiritClient;
    }

    public String checkUrlBase() {
        return checkUrlBase;
    }

    public HttpHost fiscalStoragesHost() {
        return fiscalStoragesHost;
    }

    public AsyncClient fiscalStoragesClient() {
        return fiscalStoragesClient;
    }

    public HttpHost dyngoHost() {
        return dyngoHost;
    }

    public AsyncClient dyngoClient() {
        return dyngoClient;
    }

    public HttpHost fnslinkerHost() {
        return fnslinkerHost;
    }

    public AsyncClient fnslinkerClient() {
        return fnslinkerClient;
    }
}

