package ru.yandex.ohio.backend;

import java.io.IOException;
import java.util.List;
import java.util.function.Supplier;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.FixedMultiFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.FakeAsyncConsumer;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;

public class ReceiptsHandler implements HttpAsyncRequestHandler<HttpRequest> {
    private static final DateTimeFormatter INPUT_DATE_FORMAT =
        DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");
    private static final DateTimeFormatter OUTPUT_DATE_FORMAT =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZZ").withZoneUTC();

    protected final OhioBackend server;

    public ReceiptsHandler(final OhioBackend server) {
        this.server = server;
    }

    @Override
    public FakeAsyncConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new FakeAsyncConsumer<>(request);
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        String purchaseToken = session.params().get(
            "purchase_token",
            NonEmptyValidator.INSTANCE);
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("user_requisites");
            writer.startArray();
            writer.startObject();
            writer.key("name");
            writer.value("trust_purchase_token");
            writer.key("value");
            writer.value(purchaseToken);
            writer.endObject();
            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            throw new ServiceUnavailableException(e);
        }
        AsyncClient client =
            server.darkspiritClient().adjust(session.context());
        client.execute(
            server.darkspiritHost(),
            new BasicAsyncRequestProducerGenerator(
                "/v1/receipts/search-by-payment-ids?limit=100&offset=0",
                sbw.toString(),
                ContentType.APPLICATION_JSON),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new DarkspiritCallback(
                new ResultPrinter(session),
                session,
                server));
    }

    private static JsonMap wrapReceipts(final JsonList receipts) {
        JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE, 2);
        map.put("receipts", receipts);
        return map;
    }

    // receipt url looks like:
    // https://check.yandex.ru/pdf?n=121765&fn=9960440300352394&fpd=1894990199
    private static class ReceiptInfo implements Comparable<ReceiptInfo> {
        private final String fn;    // like "9960440300352394"
        private final long fpd;     // like 1894990199
        private final long n;       // like 121765

        ReceiptInfo(final String fn, final long fpd, final long n) {
            this.fn = fn;
            this.fpd = fpd;
            this.n = n;
        }

        @Override
        public int compareTo(final ReceiptInfo other) {
            int cmp = fn.compareTo(other.fn);
            if (cmp == 0) {
                cmp = Long.compare(fpd, other.fpd);
                if (cmp == 0) {
                    cmp = Long.compare(n, other.n);
                }
            }
            return cmp;
        }

        @Override
        public String toString() {
            return fn + '/' + n + '/' + fpd;
        }
    }

    private static class ReceiptContent {
        private final String receiptType;
        private final JsonObject payments;
        private final String timestamp;
        private final String localzone;

        ReceiptContent(
            final String receiptType,
            final JsonObject payments,
            final String timestamp,
            final String localzone)
        {
            this.receiptType = receiptType;
            this.payments = payments;
            this.timestamp = timestamp;
            this.localzone = localzone;
        }
    }

    private static class FullReceipt implements Comparable<FullReceipt> {
        private final ReceiptInfo receiptInfo;
        private final ReceiptContent receiptContent;

        FullReceipt(
            final ReceiptInfo receiptInfo,
            final ReceiptContent receiptContent)
        {
            this.receiptInfo = receiptInfo;
            this.receiptContent = receiptContent;
        }

        @Override
        public int compareTo(final FullReceipt other) {
            return receiptInfo.compareTo(other.receiptInfo);
        }
    }

    private static class DarkspiritCallback
        extends FilterFutureCallback<JsonObject>
    {
        private final ProxySession session;
        private final OhioBackend server;

        DarkspiritCallback(
            final FutureCallback<? super JsonObject> callback,
            final ProxySession session,
            final OhioBackend server)
        {
            super(callback);
            this.session = session;
            this.server = server;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList items = result.get("items").asList();
                int size = items.size();
                if (size == 0) {
                    super.completed(wrapReceipts(JsonList.EMPTY));
                } else {
                    AsyncClient client =
                        server.fiscalStoragesClient().adjust(
                            session.context());
                    Supplier<? extends HttpClientContext> contextGenerator =
                        session.listener().createContextGeneratorFor(client);
                    FixedMultiFutureCallback<FullReceipt> callback =
                        new FixedMultiFutureCallback<>(
                            new ReceiptsCallback(
                                this.callback,
                                server,
                                session),
                            size);
                    for (int i = 0; i < size; ++i) {
                        JsonObject item = items.get(i);
                        ReceiptInfo receiptInfo =
                            new ReceiptInfo(
                                item.get("fn").get("sn").asString(),
                                item.get("fp").asLong(),
                                item.get("id").asLong());
                        QueryConstructor query =
                            new QueryConstructor("/v1/fiscal_storages/");
                        query.append(receiptInfo.fn);
                        StringBuilder sb = query.sb();
                        sb.append("/documents/");
                        sb.append(receiptInfo.n);
                        sb.append('/');
                        sb.append(receiptInfo.fpd);
                        client.execute(
                            server.fiscalStoragesHost(),
                            new BasicAsyncRequestProducerGenerator(
                                new String(sb)),
                            JsonAsyncTypesafeDomConsumerFactory.OK,
                            contextGenerator,
                            new ReceiptCallback(
                                callback.callback(i),
                                session,
                                receiptInfo));
                    }
                }
            } catch (BadRequestException | JsonException e) {
                session.logger().warning(
                    "Failed to parse darkspirit response: "
                    + JsonType.NORMAL.toString(result));
                failed(e);
            }
        }
    }

    private static class ReceiptCallback
        extends AbstractFilterFutureCallback<JsonObject, FullReceipt>
    {
        private final ProxySession session;
        private final ReceiptInfo receiptInfo;

        ReceiptCallback(
            final FutureCallback<? super FullReceipt> callback,
            final ProxySession session,
            final ReceiptInfo receiptInfo)
        {
            super(callback);
            this.session = session;
            this.receiptInfo = receiptInfo;
        }

        @Override
        public void completed(final JsonObject result) {
            session.logger().info(
                "Got receipt content for receipt info " + receiptInfo);
            try {
                JsonMap content = result.get("receipt_content").asMap();
                callback.completed(
                    new FullReceipt(
                        receiptInfo,
                        new ReceiptContent(
                            content.get("receipt_type").asString(),
                            content.get("payments"),
                            result.get("dt").asStringOrNull(),
                            result.get("localzone").asStringOrNull())));
            } catch (JsonException e) {
                session.logger().warning(
                    "Failed to parse fiscal storages response: "
                    + JsonType.NORMAL.toString(result));
                failed(e);
            }
        }
    }

    private static class ReceiptsCallback
        extends AbstractFilterFutureCallback<List<FullReceipt>, JsonObject>
    {
        private final OhioBackend server;
        private final ProxySession session;

        ReceiptsCallback(
            final FutureCallback<? super JsonObject> callback,
            final OhioBackend server,
            final ProxySession session)
        {
            super(callback);
            this.server = server;
            this.session = session;
        }

        @Override
        public void completed(final List<FullReceipt> result) {
            result.sort(null);
            int size = result.size();
            JsonList receipts =
                new JsonList(BasicContainerFactory.INSTANCE, size);
            try {
                for (int i = 0; i < size; ++i) {
                    FullReceipt receipt = result.get(i);
                    JsonMap receiptMap =
                        new JsonMap(BasicContainerFactory.INSTANCE, 4);

                    ReceiptInfo receiptInfo = receipt.receiptInfo;
                    QueryConstructor query =
                        new QueryConstructor(server.checkUrlBase(), false);
                    query.append("fn", receiptInfo.fn);
                    query.append("fpd", receiptInfo.fpd);
                    query.append("n", receiptInfo.n);
                    receiptMap.put("url", new JsonString(query.toString()));

                    ReceiptContent receiptContent = receipt.receiptContent;
                    receiptMap.put(
                        "receipt_type",
                        new JsonString(receiptContent.receiptType));
                    receiptMap.put("payments", receiptContent.payments);
                    try {
                        receiptMap.put(
                            "timestamp",
                            new JsonString(
                                OUTPUT_DATE_FORMAT.print(
                                    INPUT_DATE_FORMAT.withZone(
                                        DateTimeZone.forID(
                                            receiptContent.localzone))
                                        .parseMillis(
                                            receiptContent.timestamp))));
                    } catch (RuntimeException e) {
                        // TODO: stater
                        session.logger().log(
                            Level.WARNING,
                            "Failed to parse timestamp "
                            + receiptContent.timestamp
                            + " with localzone " + receiptContent.localzone,
                            e);
                    }

                    receipts.add(receiptMap);
                }
                callback.completed(wrapReceipts(receipts));
            } catch (BadRequestException e) {
                failed(e);
            }
        }
    }
}

