package ru.yandex.ohio.indexer;

import java.io.IOException;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpException;

import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.NullObjectFilter;
import ru.yandex.json.dom.OrderingContainerFactory;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.parser.uri.UriParser;

public class AddRecordHandler extends AddRecordHandlerBase {
    private static final Pattern ADMIN_UID_PATTERN =
        Pattern.compile("u'familyAdminUid':\\s*([0-9]+)");

    public AddRecordHandler(final OhioIndexer server) {
        super(server, server::commitRecordsStat);
    }

    private static JsonObject tryParseAsJsonObject(final JsonObject obj) {
        if (obj.type() != JsonObject.Type.STRING || obj.isEmpty()) {
            return obj;
        }
        try {
            return TypesafeValueContentHandler.parse(obj.asString())
                .filter(
                    NullObjectFilter.INSTANCE,
                    OrderingContainerFactory.INSTANCE);
        } catch (JsonException e) {
            return obj;
        }
    }

    private static Long extractFamilyAdminUid(final JsonMap map)
        throws JsonException
    {
        JsonMap fraudStatus = map.getMapOrNull("fraud_status");
        if (fraudStatus != null) {
            String afsTags = fraudStatus.getString("afs_tags", null);
            if (afsTags != null) {
                Matcher matcher = ADMIN_UID_PATTERN.matcher(afsTags);
                if (matcher.find()) {
                    try {
                        return Long.valueOf(matcher.group(1));
                    } catch (RuntimeException e) {
                        throw new JsonException(
                            "Malformed afs_tags <" + afsTags + '>', e);
                    }
                }
            }
        }
        return null;
    }

    private static void extractIfNotEmpty(
        final JsonWriter writer,
        final JsonObject map,
        final String field)
        throws IOException, JsonException
    {
        JsonObject value = map.get(field);
        if (value.type() == JsonObject.Type.STRING && !value.isEmpty()) {
            writer.key(field);
            value.writeValue(writer);
        }
    }

    @Override
    protected void convert(
        final List<QueryAndBody> parts,
        final UriParser uri,
        final JsonMap map,
        PrefixedLogger logger)
        throws HttpException, IOException, JsonException
    {
        String purchaseToken = map.get("purchase_token").asStringOrNull();
        if (purchaseToken == null) {
            logger.warning("No purchase_token found in " + uri);
            return;
        }
        logger = logger.addPrefix(purchaseToken);
        Long uidBoxed = map.get("passport_id").asLongOrNull();
        if (uidBoxed == null) {
            logger.warning("No passport_id found in " + uri);
            return;
        }
        logger = logger.addPrefix(uidBoxed.toString());
        long uid = uidBoxed.longValue();
        Long timestamp = parseTimestamp(map.get("dt").asStringOrNull());
        if (timestamp == null) {
            logger.warning("No dt found in " + uri);
            return;
        }
        Long terminalIdBoxed = map.get("terminal_id").asLongOrNull();
        if (terminalIdBoxed == null) {
            logger.warning("No terminal_id in " + uri);
            return;
        }
        long terminalId = terminalIdBoxed.longValue();
        Long familyAdminUid = extractFamilyAdminUid(map);
        logger.info("Extracted family admin uid: " + familyAdminUid);
        Long postauthTimestamp =
            parseTimestamp(map.get("postauth_dt").asStringOrNull());
        Long paymentTimestamp =
            parseTimestamp(map.get("payment_dt").asStringOrNull());
        Long cancelTimestamp =
            parseTimestamp(map.get("cancel_dt").asStringOrNull());
        CgiParams params = new CgiParams(uri.queryParser());
        long sequenceNumber = params.getLong("seqNo");
        long messageWriteTime = params.getLong("message-write-time");
        Long firmId = map.get("firm_id").asLongOrNull();
        long serviceId = map.get("service_id").asLong();
        String userAccount = map.get("user_account").asStringOrNull();
        String userPhone = map.get("user_phone").asStringOrNull();
        String userEmail = map.get("user_email").asStringOrNull();
        String paymentMethod = map.get("payment_method").asString();
        int idx = paymentMethod.indexOf('-');
        String normalizedPaymentMethod;
        if (idx == -1) {
            normalizedPaymentMethod = paymentMethod;
        } else {
            normalizedPaymentMethod = paymentMethod.substring(0, idx);
        }
        Long paymentMethodId = map.get("payment_method_id").asLongOrNull();
        String paymentMode = map.get("payment_mode").asStringOrNull();
        String paymentType = map.get("payment_type").asString();
        String paymentStatus = map.get("payment_status").asStringOrNull();
        double amount = map.get("amount").asDouble();
        Double postauthAmount = map.get("postauth_amount").asDoubleOrNull();
        Double cashbackAmount = map.get("cashback_amount").asDoubleOrNull();
        String cashbackParentId =
            map.get("cashback_parent_id").asStringOrNull();
        String currency = map.get("currency").asString();
        String trustGroupId = map.get("trust_group_id").asStringOrNull();
        String trustPaymentId = map.get("trust_payment_id").asString();
        String compositePaymentId =
            map.get("composite_payment_id").asStringOrNull();
        JsonObject refunds =
            map.get("refunds").filter(
                NullObjectFilter.INSTANCE,
                OrderingContainerFactory.INSTANCE);
        JsonObject rows =
            map.get("rows").filter(
                NullObjectFilter.INSTANCE,
                OrderingContainerFactory.INSTANCE);
        JsonObject products =
            map.get("products").filter(
                NullObjectFilter.INSTANCE,
                OrderingContainerFactory.INSTANCE);
        JsonObject compositeComponents =
            map.get("composite_components").filter(
                NullObjectFilter.INSTANCE,
                OrderingContainerFactory.INSTANCE);
        JsonObject developerPayload =
            tryParseAsJsonObject(map.get("developer_payload"));

        Long queryUid = uid;
        Long initiatorUid = null;
        Long sponsorUid;
        if (familyAdminUid == null || familyAdminUid.longValue() == uid) {
            sponsorUid = null;
        } else {
            sponsorUid = familyAdminUid;
        }

        for (int i = 0; i < 2 && queryUid != null; ++i) {
            uid = queryUid.longValue();
            QueryConstructor query =
                new QueryConstructor("/update?service=ohio_index");
            query.append("prefix", uid);
            query.append("seqNo", sequenceNumber);
            query.append("service_id", serviceId);
            query.append("terminal_id", terminalId);
            query.append("purchase_token", purchaseToken);

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(uid);
                writer.key("AddIfNotExists");
                writer.value(true);
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writer.key("data_format_version");
                writer.value(9L);
                writer.key("order_revision");
                writer.startObject();
                writer.key("function");
                writer.value("inc");
                writer.endObject();
                writer.key("purchase_token");
                writer.value(purchaseToken);
                writer.key("uid");
                writer.value(uid);
                if (sponsorUid != null) {
                    writer.key("sponsor_uid");
                    writer.value(sponsorUid.longValue());
                    query.append("sponsor_uid", sponsorUid.longValue());
                }
                if (initiatorUid != null) {
                    writer.key("initiator_uid");
                    writer.value(initiatorUid.longValue());
                    query.append("initiator_uid", initiatorUid.longValue());
                }
                writer.key("timestamp");
                writer.value(timestamp.longValue());
                writer.key("update_timestamp");
                writer.value(messageWriteTime);
                if (postauthTimestamp != null) {
                    writer.key("postauth_timestamp");
                    writer.value(postauthTimestamp.longValue());
                }
                if (paymentTimestamp != null) {
                    writer.key("payment_timestamp");
                    writer.value(paymentTimestamp.longValue());
                }
                if (cancelTimestamp != null) {
                    writer.key("cancel_timestamp");
                    writer.value(cancelTimestamp.longValue());
                }
                writer.key("sequence_number");
                writer.startObject();
                writer.key("function");
                writer.value("default");
                writer.key("args");
                writer.startArray();
                writer.value(sequenceNumber);
                writer.endArray();
                writer.endObject();
                if (firmId != null) {
                    long value = firmId.longValue();
                    query.append("firm_id", value);
                    writer.key("firm_id");
                    writer.value(value);
                }
                writer.key("service_id");
                writer.value(serviceId);
                writer.key("terminal_id");
                writer.value(terminalId);
                if (userAccount != null) {
                    writer.key("user_account");
                    writer.value(userAccount);
                }
                if (userPhone != null) {
                    writer.key("user_phone");
                    writer.value(userPhone);
                }
                if (userEmail != null) {
                    writer.key("user_email");
                    writer.value(userEmail);
                }
                writer.key("payment_method");
                writer.value(paymentMethod);
                writer.key("normalized_payment_method");
                writer.value(normalizedPaymentMethod);
                if (paymentMethodId != null) {
                    writer.key("payment_method_id");
                    writer.value(paymentMethodId.longValue());
                }
                if (paymentMode != null) {
                    writer.key("payment_mode");
                    writer.value(paymentMode);
                }
                writer.key("payment_type");
                writer.value(paymentType);
                if (paymentStatus != null) {
                    query.append("payment_status", paymentStatus);

                    writer.key("payment_status");
                    writer.startObject();
                    writer.key("function");
                    writer.value("make_set");
                    writer.key("args");
                    writer.startArray();
                    writer.startObject();
                    writer.key("function");
                    writer.value("get");
                    writer.key("args");
                    writer.startArray();
                    writer.value("payment_status");
                    writer.endArray();
                    writer.endObject();
                    writer.value(paymentStatus);
                    writer.endArray();
                    writer.endObject();

                    writer.key("last_payment_status");
                    writer.value(paymentStatus);
                }
                writer.key("amount");
                writer.value(amount);
                if (postauthAmount != null) {
                    writer.key("postauth_amount");
                    writer.value(postauthAmount.doubleValue());
                }
                if (cashbackAmount != null) {
                    writer.key("cashback_amount");
                    writer.value(cashbackAmount.doubleValue());
                }
                if (cashbackParentId != null) {
                    writer.key("cashback_parent_id");
                    writer.value(cashbackParentId);
                }
                writer.key("currency");
                writer.value(currency);
                if (trustGroupId != null) {
                    writer.key("trust_group_id");
                    writer.value(trustGroupId);
                }
                writer.key("trust_payment_id");
                writer.value(trustPaymentId);
                if (compositePaymentId != null) {
                    writer.key("composite_payment_id");
                    writer.value(compositePaymentId);
                }
                if (refunds != JsonNull.INSTANCE) {
                    writer.key("refunds");
                    writer.value(JsonType.NORMAL.toString(refunds));
                }
                if (rows != JsonNull.INSTANCE) {
                    writer.key("rows");
                    writer.value(JsonType.NORMAL.toString(rows));
                }
                if (!products.isEmpty()) {
                    writer.key("products");
                    writer.value(JsonType.NORMAL.toString(products));
                }
                if (!compositeComponents.isEmpty()) {
                    writer.key("composite_components");
                    writer.value(JsonType.NORMAL.toString(compositeComponents));
                }
                if (!developerPayload.isEmpty()) {
                    writer.key("developer_payload");
                    writer.value(JsonType.NORMAL.toString(developerPayload));
                    if (developerPayload.type() == JsonObject.Type.MAP) {
                        extractIfNotEmpty(
                            writer,
                            developerPayload,
                            "points_type");
                        extractIfNotEmpty(
                            writer,
                            developerPayload,
                            "order_trust_payment_id");
                        extractIfNotEmpty(
                            writer,
                            developerPayload,
                            "cashback_service");
                    }
                }
                writer.endObject();
                writer.endArray();
                writer.endObject();
            }
            String queryString = query.toString();
            logger.info("Record converted successfully to " + queryString);
            parts.add(new QueryAndBody(queryString, sbw.toString(), uid));
            initiatorUid = queryUid;
            queryUid = sponsorUid;
            sponsorUid = null;
        }
    }
}

