package ru.yandex.ohio.indexer;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.MimeTokenStream;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.charset.Decoder;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.function.Processable;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpResponseSendingCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.ByteArrayProcessableWithContentTypeAsyncConsumer;
import ru.yandex.http.util.nio.NByteArrayEntityGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.ByteArrayInputStreamFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.parser.uri.UriParser;

public abstract class AddRecordHandlerBase
    implements HttpAsyncRequestHandler<ByteArrayProcessableWithContentType>
{
    private static final DateTimeFormatter[] DATE_FORMATS =
        new DateTimeFormatter[] {
            DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZ"),
            DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZZ")
                .withZoneUTC(),
            DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSSSSZZ")
                .withZoneUTC(),
            DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSSSSSSSZZ")
                .withZoneUTC(),
            DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZZ").withZoneUTC(),
            DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss'Z'").withZoneUTC()
        };

    private final OhioIndexer server;
    private final Consumer<OhioIndexer.RecordsStat> recordsStatsConsumer;

    protected AddRecordHandlerBase(
        final OhioIndexer server,
        final Consumer<OhioIndexer.RecordsStat> recordsStatsConsumer)
    {
        this.server = server;
        this.recordsStatsConsumer = recordsStatsConsumer;
    }

    public static Long parseTimestamp(final String timestamp)
        throws BadRequestException
    {
        if (timestamp == null) {
            return null;
        } else {
            Throwable error = null;
            for (DateTimeFormatter format: DATE_FORMATS) {
                try {
                    return format.parseMillis(timestamp);
                } catch (Throwable t) {
                    if (error == null) {
                        error = t;
                    }
                }
            }
            throw new BadRequestException(
                "Failed to parse timestamp <" + timestamp + '>',
                error);
        }
    }

    @Override
    public HttpAsyncRequestConsumer<ByteArrayProcessableWithContentType>
    processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new ByteArrayProcessableWithContentTypeAsyncConsumer();
            }
        }
        throw new BadRequestException("Payload expected");
    }

    @Override
    public void handle(
        final ByteArrayProcessableWithContentType payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        List<QueryAndBody> parts = new ArrayList<>(2);
        if ("multipart/mixed".equals(payload.contentType().getMimeType())) {
            handleMultipart(session, payload, parts);
        } else {
            handleSinglepart(session, payload.data(), parts);
        }
        int size = parts.size();
        if (size > 0) {
            AsyncClient client =
                server.producerStoreClient().adjust(session.context());
            try {
                BasicAsyncRequestProducerGenerator producerGenerator;
                if (size == 1) {
                    QueryAndBody queryAndBody = parts.get(0);
                    producerGenerator =
                        new BasicAsyncRequestProducerGenerator(
                            queryAndBody.query,
                            new NByteArrayEntityGenerator(
                                queryAndBody.body,
                                ContentType.APPLICATION_JSON));
                } else {
                    MultipartEntityBuilder builder =
                        MultipartEntityBuilder.create();
                    builder.setMimeSubtype("mixed");
                    for (int i = 0; i < size; ++i) {
                        QueryAndBody queryAndBody = parts.get(i);
                        builder.addPart(
                            FormBodyPartBuilder
                                .create()
                                .addField(
                                    YandexHeaders.URI,
                                    queryAndBody.query)
                                .addField(
                                    "prefix",
                                    Long.toString(queryAndBody.uid))
                                .setBody(
                                    new StringBody(
                                        queryAndBody.body,
                                        ContentType.APPLICATION_JSON))
                                .setName("message.bin")
                                .build());
                    }
                    producerGenerator =
                        new BasicAsyncRequestProducerGenerator(
                            session.uri().toString() + "&service=ohio_index",
                            builder.build());
                }
                client.execute(
                    server.producerStoreHost(),
                    producerGenerator,
                    BasicAsyncResponseConsumerFactory.ANY_GOOD,
                    session.listener().createContextGeneratorFor(client),
                    new HttpResponseSendingCallback(session));
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            }
        } else {
            session.response(HttpStatus.SC_OK);
        }
    }

    private void handleSinglepart(
        final ProxySession session,
        final Processable<byte[]> data,
        final List<QueryAndBody> parts)
    {
        BasicGenericConsumer<JsonObject, JsonException> consumer =
            new BasicGenericConsumer<>();
        JsonParser parser =
            TypesafeValueContentHandler.prepareParser(consumer);
        Decoder decoder = new Decoder(StandardCharsets.UTF_8);
        int initialPartsSize = parts.size();
        try {
            data.processWith(decoder);
            decoder.processWith(parser);
            decoder = null;
            parser.eof();
            convert(
                parts,
                session.uri(),
                consumer.get().asMap(),
                session.logger());
        } catch (HttpException | IOException | JsonException e) {
            recordsStatsConsumer.accept(OhioIndexer.RecordsStat.MALFORMED);
            session.logger().log(
                Level.WARNING,
                "Malformed record: "
                + JsonType.NORMAL.toString(consumer.get()),
                e);
            return;
        }
        if (initialPartsSize == parts.size()) {
            recordsStatsConsumer.accept(OhioIndexer.RecordsStat.INCOMPLETE);
        } else {
            recordsStatsConsumer.accept(OhioIndexer.RecordsStat.OK);
        }
    }

    private void handleMultipart(
        final ProxySession session,
        final ByteArrayProcessableWithContentType payload,
        final List<QueryAndBody> parts)
        throws HttpException
    {
        BasicGenericConsumer<JsonObject, JsonException> consumer =
            new BasicGenericConsumer<>();
        JsonParser parser =
            TypesafeValueContentHandler.prepareParser(consumer);
        Decoder decoder = new Decoder(StandardCharsets.UTF_8);
        MimeTokenStream stream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        stream.parseHeadless(
            payload.data().processWith(
                ByteArrayInputStreamFactory.INSTANCE),
            payload.contentTypeHeader().getValue());

        long incomplete = 0L;
        long malformed = 0L;
        long total = 0L;
        try {
            EntityState state = stream.getState();
            String uri = null;
            while (state != EntityState.T_END_OF_STREAM) {
                switch (state) {
                    case T_FIELD:
                        Field field = stream.getField();
                        if (YandexHeaders.URI.equals(field.getName())) {
                            uri = field.getBody();
                        }
                        break;
                    case T_BODY:
                        ++total;
                        DecodableByteArrayOutputStream data =
                            IOStreamUtils.consume(
                                stream.getDecodedInputStream());
                        if (uri == null) {
                            throw new BadRequestException(
                                "No uri found for part");
                        }
                        try {
                            data.processWith(decoder);
                            parser.reset();
                            decoder.processWith(parser);
                            parser.eof();
                            int initialPartsSize = parts.size();
                            convert(
                                parts,
                                new UriParser(uri),
                                consumer.get().asMap(),
                                session.logger());
                            if (initialPartsSize == parts.size()) {
                                ++incomplete;
                            }
                        } catch (HttpException | IOException | JsonException e) {
                            ++malformed;
                            session.logger().log(
                                Level.WARNING,
                                "Malformed record for uri " + uri
                                + ':' + ' '
                                + JsonType.NORMAL.toString(consumer.get()),
                                e);
                        }
                        uri = null;
                        break;
                    default:
                        break;
                }
                state = stream.next();
            }
        } catch (IOException | MimeException e) {
            throw new BadRequestException(e);
        }
        recordsStatsConsumer.accept(
            new OhioIndexer.RecordsStat(incomplete, malformed, total));
        if (incomplete + malformed >= total) {
            session.logger().warning(
                "All parts either incomplete or malformed");
        }
    }

    protected abstract void convert(
        final List<QueryAndBody> parts,
        final UriParser uri,
        final JsonMap map,
        final PrefixedLogger logger)
        throws HttpException, IOException, JsonException;

    protected static class QueryAndBody {
        private final String query;
        private final String body;
        private final long uid;

        QueryAndBody(final String query, final String body, final long uid) {
            this.query = query;
            this.body = body;
            this.uid = uid;
        }
    }
}

