package ru.yandex.ohio.indexer;

import java.io.IOException;
import java.util.List;

import org.apache.http.HttpException;

import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.NullObjectFilter;
import ru.yandex.json.dom.OrderingContainerFactory;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.parser.uri.UriParser;

public class AddYandexPayRecordHandler extends AddRecordHandlerBase {
    public AddYandexPayRecordHandler(final OhioIndexer server) {
        super(server, server::commitYandexPayRecordsStat);
    }

    @Override
    protected void convert(
        final List<QueryAndBody> parts,
        final UriParser uri,
        final JsonMap map,
        PrefixedLogger logger)
        throws HttpException, IOException, JsonException
    {
        String messageId = map.get("message_id").asStringOrNull();
        if (messageId == null) {
            logger.warning("No message_id found in " + uri);
            return;
        }
        logger.info("Processing record for message_id " + messageId);
        Long uidBoxed = map.get("uid").asLongOrNull();
        if (uidBoxed == null) {
            logger.warning("No uid found in " + uri);
            return;
        }
        logger = logger.addPrefix(uidBoxed.toString());
        long uid = uidBoxed.longValue();
        Long created = parseTimestamp(map.get("created").asStringOrNull());
        if (created == null) {
            logger.warning("Field 'created' not found in " + uri);
            return;
        }
        Long updated = parseTimestamp(map.get("updated").asStringOrNull());
        String status = map.get("status").asStringOrNull();
        String currency = map.get("currency").asString();
        double amount = map.get("amount").asDouble();
        JsonMap merchant = map.get("merchant").asMapOrNull();
        JsonObject items =
            map.get("items").filter(
                NullObjectFilter.INSTANCE,
                OrderingContainerFactory.INSTANCE);
        String plusPurchaseToken =
            map.get("plus_purchase_token").asStringOrNull();

        CgiParams params = new CgiParams(uri.queryParser());
        long sequenceNumber = params.getLong("seqNo");

        QueryConstructor query =
            new QueryConstructor("/update?service=ohio_index");
        query.append("prefix", uid);
        query.append("seqNo", sequenceNumber);
        query.append("messageId", messageId);
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = new DollarJsonWriter(sbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(uid);
            writer.key("AddIfNotExists");
            writer.value(true);
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            writer.key("source");
            writer.value("yandexpay");
            writer.key("data_format_version");
            writer.value(7L);
            writer.key("order_revision");
            writer.startObject();
            writer.key("function");
            writer.value("inc");
            writer.endObject();
            writer.key("purchase_token");
            writer.value(messageId);
            writer.key("uid");
            writer.value(uid);
            writer.key("timestamp");
            writer.value(created.longValue());
            if (updated != null) {
                writer.key("update_timestamp");
                writer.value(updated.longValue());
            }
            writer.key("sequence_number");
            writer.startObject();
            writer.key("function");
            writer.value("default");
            writer.key("args");
            writer.startArray();
            writer.value(sequenceNumber);
            writer.endArray();
            writer.endObject();
            if (status != null) {
                query.append("status", status);

                writer.key("payment_status");
                writer.startObject();
                writer.key("function");
                writer.value("make_set");
                writer.key("args");
                writer.startArray();
                writer.startObject();
                writer.key("function");
                writer.value("get");
                writer.key("args");
                writer.startArray();
                writer.value("payment_status");
                writer.endArray();
                writer.endObject();
                writer.value(status);
                writer.endArray();
                writer.endObject();

                writer.key("last_payment_status");
                writer.value(status);
            }
            writer.key("amount");
            writer.value(amount);
            writer.key("currency");
            writer.value(currency);
            if (merchant != null) {
                String id = merchant.get("id").asStringOrNull();
                if (id != null) {
                    writer.key("merchant_id");
                    writer.value(id);
                }
                String gatewayUrl =
                    merchant.get("gateway_url").asStringOrNull();
                if (gatewayUrl != null) {
                    writer.key("gateway_url");
                    writer.value(gatewayUrl);
                }
                String gatewayName =
                    merchant.get("gateway_name").asStringOrNull();
                if (gatewayName != null) {
                    writer.key("gateway_name");
                    writer.value(gatewayName);
                }
            }
            if (!items.isEmpty()) {
                writer.key("yandexpay_items");
                writer.value(JsonType.NORMAL.toString(items));
            }
            if (plusPurchaseToken != null) {
                writer.key("plus_purchase_token");
                writer.value(plusPurchaseToken);
            }
            writer.endObject();
            writer.endArray();
            writer.endObject();
        }
        String queryString = query.toString();
        logger.info("Record converted successfully to " + queryString);
        parts.add(new QueryAndBody(queryString, sbw.toString(), uid));
    }
}

