package ru.yandex.ohio.indexer;

import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.Cancellable;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.collection.IntPair;
import ru.yandex.compress.GzipInputStream;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.FakeAsyncConsumer;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.NullObjectFilter;
import ru.yandex.json.dom.OrderingContainerFactory;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;

public class ImportFileHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final DateTimeFormatter FALLBACK_DATE_FORMAT =
        DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZZ").withZoneUTC();

    private final OhioIndexer server;

    public ImportFileHandler(final OhioIndexer server) {
        this.server = server;
    }

    @Override
    public FakeAsyncConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new FakeAsyncConsumer<>(request);
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        new Context(server, new BasicProxySession(server, exchange, context))
            .processNextBatch();
    }

    private static class Context implements Cancellable {
        private final List<String> batch = new ArrayList<>();
        private final OhioIndexer server;
        private final ProxySession session;
        private final int batchSize;
        private final BufferedReader reader;
        private volatile int linesProcessed;

        Context(final OhioIndexer server, final ProxySession session)
            throws HttpException
        {
            this.server = server;
            this.session = session;
            batchSize = session.params().getInt("batch-size");
            File file = session.params().getInputFile("file");
            boolean ungzip = session.params().getBoolean("ungzip", false);
            linesProcessed = session.params().getInt("offset", 0);
            try {
                reader =
                    new BufferedReader(
                        new InputStreamReader(
                            openFile(file, ungzip),
                            StandardCharsets.UTF_8),
                        1 << 20);
                try {
                    for (int i = 0; i < linesProcessed; ++i) {
                        if (reader.readLine() == null) {
                            throw new EOFException(
                                "Unexpected EOF after line " + i);
                        }
                    }
                } catch (IOException e) {
                    try {
                        reader.close();
                    } catch (IOException ex) {
                        e.addSuppressed(ex);
                    }
                    throw new ServiceUnavailableException(e);
                }
            } catch (IOException e) {
                throw new ServiceUnavailableException(e);
            }
            session.subscribeForCancellation(this);
        }

        private static InputStream openFile(
            final File file,
            final boolean ungzip)
            throws IOException
        {
            InputStream in = new FileInputStream(file);
            if (ungzip) {
                try {
                    return new GzipInputStream(in, 1 << 20);
                } catch (IOException e) {
                    try {
                        in.close();
                    } catch (IOException ex) {
                        e.addSuppressed(ex);
                    }
                    throw e;
                }
            } else {
                return in;
            }
        }

        @Override
        public boolean cancel() {
            session.logger().info("Closing reader");
            try {
                reader.close();
            } catch (IOException e) {
                session.logger().log(Level.WARNING, "Reader close failed", e);
            }
            return true;
        }

        public void processNextBatch() {
            batch.clear();
            int i = 0;
            try {
                for (; i < batchSize; ++i) {
                    String line = reader.readLine();
                    if (line == null) {
                        break;
                    }
                    batch.add(line);
                }
            } catch (IOException e) {
                session.handleException(
                    new ServiceUnavailableException(
                        "Read failed at line " + (linesProcessed + i),
                        e));
                return;
            }
            session.logger().info(
                batch.size() + " lines read from position " + linesProcessed);
            if (batch.isEmpty()) {
                session.response(
                    HttpStatus.SC_OK,
                    linesProcessed + " lines processed");
                return;
            }
            sendBatch();
        }

        private void sendBatch() {
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMimeSubtype("mixed");
            int size = batch.size();
            int skipped = 0;
            for (int i = 0; i < size; ++i) {
                String line = batch.get(i);
                try {
                    String reason = addLine(builder, line, linesProcessed + i);
                    if (reason != null) {
                        ++skipped;
                        session.logger().info(
                            "Skipping line #" + (linesProcessed + i)
                            + " because of " + reason + ':' + ' ' + line);
                    }
                } catch (HttpException | IOException | JsonException e) {
                    session.handleException(
                        new ServiceUnavailableException(
                            "Failed to process line #" + (linesProcessed + i)
                            + ':' + ' ' + line,
                            e));
                    return;
                }
            }

            if (skipped == size) {
                session.logger().info("All lines skipped, moving forward");
                new Callback(this).completed(null);
                return;
            }

            try {
                server.producerStoreClient().execute(
                    server.producerStoreHost(),
                    new BasicAsyncRequestProducerGenerator(
                        "/add?service=ohio_index&batch-size=" + size
                        + "&start-line=" + linesProcessed,
                        builder.build()),
                    StatusCodeAsyncConsumerFactory.ANY_GOOD,
                    new IdempotentFutureCallback<>(new Callback(this)));
            } catch (IOException e) {
                session.handleException(
                    new ServiceUnavailableException(
                        "Failed to send batch of size " + size
                        + " starting from line " + linesProcessed,
                        e));
                return;
            }
        }

        private static Long parseTimestamp(final String timestamp)
            throws HttpException
        {
            try {
                return AddYandexPayRecordHandler.parseTimestamp(timestamp);
            } catch (Throwable t) {
                try {
                    return FALLBACK_DATE_FORMAT.parseMillis(timestamp);
                } catch (Throwable tt) {
                    t.addSuppressed(tt);
                    throw t;
                }
            }
        }

        private static String addLine(
            final MultipartEntityBuilder builder,
            final String line,
            final int offset)
            throws HttpException, IOException, JsonException
        {
            JsonMap map = TypesafeValueContentHandler.parse(line).asMap();

            String messageId = map.get("message_id").asStringOrNull();
            if (messageId == null) {
                return "null message_id";
            }
            long uid = map.get("uid").asLong();
            Long created = parseTimestamp(map.get("created").asStringOrNull());
            if (created == null) {
                return "null created";
            }
            Long updated = parseTimestamp(map.get("updated").asStringOrNull());
            String currency = map.get("currency").asString();
            double amount = map.get("amount").asDouble();
            String status = map.get("status").asStringOrNull();
            if (status == null) {
                return "null status";
            }
            switch (status) {
                case "new":
                    status = "created";
                    break;
                case "hold":
                    status = "captured";
                    break;
                case "fail":
                    status = "failed";
                    break;
                case "success":
                    status = "charged";
                    break;
                case "reverse":
                case "chargeback":
                    status = "cancelled";
                    break;
                case "refund":
                    if (amount > 0d) {
                        status = "charged";
                    } else {
                        status = "cancelled";
                    }
                    break;
                default:
                    return "unknown status <" + status + '>';
            }
            String merchantId = map.get("merchant_id").asStringOrNull();
            JsonMap data = map.get("data").asMap();
            String gatewayName = data.get("merchant_name").asStringOrNull();
            String gatewayUrl = data.get("merchant_url").asStringOrNull();
            JsonObject items =
                Objects.requireNonNullElse(
                    data.get("order_basket").asMapOrNull(),
                    JsonMap.EMPTY)
                    .get("items").filter(
                        NullObjectFilter.INSTANCE,
                        OrderingContainerFactory.INSTANCE);
            long sequenceNumber = 1642792500000000L + offset;

            QueryConstructor query =
                new QueryConstructor("/add?service=ohio_index");
            query.append("prefix", uid);
            query.append("timestamp", created.longValue());
            query.append("message_id", messageId);
            query.append("offset", offset);

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(uid);
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writer.key("source");
                writer.value("yandexpay");
                writer.key("data_format_version");
                writer.value(8L);
                writer.key("order_revision");
                writer.value(1L);
                writer.key("purchase_token");
                writer.value(messageId);
                writer.key("uid");
                writer.value(uid);
                writer.key("timestamp");
                writer.value(created.longValue());
                if (updated != null) {
                    writer.key("update_timestamp");
                    writer.value(updated.longValue());
                }
                writer.key("sequence_number");
                writer.value(sequenceNumber);
                if (status != null) {
                    query.append("status", status);

                    writer.key("payment_status");
                    writer.value(status);

                    writer.key("last_payment_status");
                    writer.value(status);
                }
                writer.key("amount");
                writer.value(amount);
                writer.key("currency");
                writer.value(currency);
                if (merchantId != null) {
                    writer.key("merchant_id");
                    writer.value(merchantId);
                }
                if (gatewayUrl != null) {
                    writer.key("gateway_url");
                    writer.value(gatewayUrl);
                }
                if (gatewayName != null) {
                    writer.key("gateway_name");
                    writer.value(gatewayName);
                }
                if (!items.isEmpty()) {
                    writer.key("yandexpay_items");
                    writer.value(JsonType.NORMAL.toString(items));
                }
                writer.endObject();
                writer.endArray();
                writer.endObject();
            }

            builder.addPart(
                FormBodyPartBuilder
                    .create()
                    .addField(YandexHeaders.URI, query.toString())
                    .addField("prefix", Long.toString(uid))
                    .setBody(
                        new StringBody(
                            sbw.toString(),
                            ContentType.TEXT_PLAIN.withCharset(
                                StandardCharsets.UTF_8)))
                    .setName("message.json")
                    .build());
            return null;
        }
    }

    private static class Callback implements FutureCallback<IntPair<Void>> {
        private final Context context;

        Callback(final Context context) {
            this.context = context;
        }

        @Override
        public void cancelled() {
            context.session.logger().warning(
                "Request cancelled, total lines processed so far: "
                + context.linesProcessed);
        }

        @Override
        public void completed(final IntPair<Void> result) {
            int size = context.batch.size();
            context.linesProcessed += size;
            context.session.logger().info(
                "Batch processed, batch size: " + size
                + ", total lines processed: " + context.linesProcessed);
            context.processNextBatch();
        }

        @Override
        public void failed(final Exception e) {
            context.session.logger().log(
                Level.WARNING,
                "Failed to process batch of size " + context.batch.size()
                + " starting at line " + context.linesProcessed,
                e);
            context.sendBatch();
        }
    }
}

