package ru.yandex.ohio.indexer;

import java.io.IOException;
import java.util.function.BinaryOperator;

import org.apache.http.HttpHost;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ConcurrentAccumulator;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.ohio.indexer.config.ImmutableOhioIndexerConfig;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.stater.AlertThresholds;
import ru.yandex.stater.GolovanAlertsConfig;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class OhioIndexer extends HttpProxy<ImmutableOhioIndexerConfig> {
    private final RecordsStater recordsStater = new RecordsStater("");
    private final RecordsStater yandexPayRecordsStater =
        new RecordsStater("yandexpay-");
    private final HttpHost producerStoreHost;
    private final AsyncClient producerStoreClient;

    public OhioIndexer(final ImmutableOhioIndexerConfig config)
        throws IOException
    {
        super(config);
        ImmutableHttpHostConfig producerStoreConfig =
            config.producerStoreConfig();
        producerStoreHost = producerStoreConfig.host();
        producerStoreClient = client("ProducerStore", producerStoreConfig);

        register(
            new Pattern<>("/add-record", false),
            new AddRecordHandler(this));

        register(
            new Pattern<>("/add-yandexpay-record", false),
            new AddYandexPayRecordHandler(this));

        register(
            new Pattern<>("/import-file", false),
            new ImportFileHandler(this));

        registerStater(recordsStater);

        registerStater(yandexPayRecordsStater);
    }

    public HttpHost producerStoreHost() {
        return producerStoreHost;
    }

    public AsyncClient producerStoreClient() {
        return producerStoreClient;
    }

    public void commitRecordsStat(final RecordsStat stat) {
        recordsStater.accumulator.accept(stat);
    }

    public void commitYandexPayRecordsStat(final RecordsStat stat) {
        yandexPayRecordsStater.accumulator.accept(stat);
    }

    private static class RecordsStater implements Stater {
        private final ConcurrentAccumulator<RecordsStat> accumulator =
            new ConcurrentAccumulator<>(
                RecordsAccumulator.INSTANCE,
                RecordsStat.ZERO);
        private final String prefix;

        RecordsStater(final String prefix) {
            this.prefix = prefix;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            RecordsStat stat = accumulator.get();
            statsConsumer.stat(
                prefix + "incomplete-records_dmmm",
                stat.incomplete);
            statsConsumer.stat(
                prefix + "malformed-records_dmmm",stat.malformed);
            statsConsumer.stat(
                prefix + "total-records_dmmm",
                stat.count);
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();

            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);
            GolovanChart records = new GolovanChart(
                prefix + "records-processed",
                " records processed",
                false,
                true,
                0d);
            records.addSplitSignal(
                config,
                statsPrefix + prefix + "total-records_dmmm",
                0,
                false,
                false);
            group.addChart(records);

            GolovanChart malformed = new GolovanChart(
                prefix + "malformed-records",
                " malformed records",
                false,
                true,
                0d);
            malformed.addSplitSignal(
                config,
                statsPrefix + prefix + "malformed-records_dmmm",
                0,
                false,
                false);
            group.addChart(malformed);

            GolovanChart incomplete = new GolovanChart(
                prefix + "incomplete-records",
                " incomplete records (%)",
                false,
                true,
                0d);
            incomplete.addSplitSignal(
                config,
                "perc(" + statsPrefix + prefix + "incomplete-records_dmmm,"
                + statsPrefix + prefix + "total-records_dmmm)",
                1,
                true,
                false);
            group.addChart(incomplete);

            panel.addCharts("records", null, group);
        }

        @Override
        public void addToAlertsConfig(
            final IniConfig alertsConfig,
            final ImmutableGolovanPanelConfig panelConfig,
            final String statsPrefix)
            throws BadRequestException
        {
            ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
            alerts.createAlert(
                alertsConfig,
                alerts.module()
                + '-'
                + GolovanAlertsConfig.clearAlertName(prefix)
                + "malformed-records",
                statsPrefix + prefix + "malformed-records_dmmm",
                new AlertThresholds(0.5d, null, null));
        }
    }

    public static class RecordsStat {
        public static final RecordsStat ZERO = new RecordsStat(0L, 0L, 0L);
        public static final RecordsStat OK =
            new RecordsStat(0L, 0L, 1L);
        public static final RecordsStat MALFORMED =
            new RecordsStat(0L, 1L, 1L);
        public static final RecordsStat INCOMPLETE =
            new RecordsStat(1L, 0L, 1L);

        private final long incomplete;
        private final long malformed;
        private final long count;

        public RecordsStat(
            final long incomplete,
            final long malformed,
            final long count)
        {
            this.incomplete = incomplete;
            this.malformed = malformed;
            this.count = count;
        }
    }

    private enum RecordsAccumulator implements BinaryOperator<RecordsStat> {
        INSTANCE;

        @Override
        public RecordsStat apply(
            final RecordsStat lhs,
            final RecordsStat rhs)
        {
            return new RecordsStat(
                lhs.incomplete + rhs.incomplete,
                lhs.malformed + rhs.malformed,
                lhs.count + rhs.count);
        }
    }
}

