from tornado.netutil import bind_sockets
from tornado import ioloop
import socket
import threading
from tornado import gen, web, httpclient
from json import dumps
from tornado.httpserver import HTTPServer
import time
import json

from database import DB
import settings


class HttpServer(HTTPServer):
    def bind(self, port, address=None, family=socket.AF_UNSPEC, backlog=128,
             reuse_port=False):

        sockets = bind_sockets(port, address=address, family=family,
                               backlog=backlog, reuse_port=reuse_port)
        if self._started:
            self.add_sockets(sockets)
        else:
            self._pending_sockets.extend(sockets)


class WebThread(threading.Thread):
    def __init__(self, name):
        threading.Thread.__init__(self, name=name)
        self.io_loop = ioloop.IOLoop()

    def run(self):
        application = get_application()
        http_server = HttpServer(application)
        http_server.bind(port=settings.conf['Daemon']['Port'], reuse_port=True)
        http_server.start()

        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'message': 'Starting thread {}'.format(self.name)
                   }
        settings.service_logger.info(dumps(log_msg))

        self.io_loop.start()

    def stop(self):
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'message': 'Stopping thread {}'.format(self.name)
                   }
        settings.service_logger.info(dumps(log_msg))
        self.io_loop.stop()


class PingHandler(web.RequestHandler):
    @gen.coroutine
    def get(self):
        self.set_header('Content-Type', 'text/plain')
        self.write("pong")
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': '/ping',
                   'response': 'pong'
                   }
        settings.access_logger.info(dumps(log_msg))
        self.finish()


class MultipleMids(Exception):
    pass


class NoMids(Exception):
    pass


def fetchDBByOraMid(ora_mid, handler):
    db = DB(settings.conf)
    data = db.fetchall('SELECT * FROM {table_name} WHERE ora_mid=%(ora_mid)s'.format(
        table_name=settings.conf['DB']['TableName']), {'ora_mid': int(ora_mid)})

    if len(data) == 1:
        res = {'uid': data[0][2], 'pg_mid': data[0][1]}

        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': handler,
                   'response': dumps(res)
                   }
        settings.access_logger.info(dumps(log_msg))
        return res
    elif len(data) > 1:
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': handler,
                   'error': "DB error, ora_mid={ora_mid} repeats several times".format(ora_mid=ora_mid)
                   }
        settings.error_logger.error(dumps(log_msg))
        raise MultipleMids
    else:
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': handler,
                   'error': "DB error, ora_mid={ora_mid} not found".format(ora_mid=ora_mid)
                   }
        settings.error_logger.info(dumps(log_msg))
        raise NoMids


class UidByMid(web.RequestHandler):
    @gen.coroutine
    def get(self):
        ora_mid = self.get_argument('mid')
        try:
            data = fetchDBByOraMid(ora_mid, '/uid_by_mid')
            self.set_header('Content-Type', 'application/json')
            self.write(dumps({'uid': data['uid']}))
            self.finish()
        except MultipleMids:
            raise web.HTTPError(500)
        except NoMids:
            raise web.HTTPError(404)


def parse(resp, default, log_func):
    try:
        obj = json.loads(resp)
        if 'envelopes' in obj:
            if len(obj['envelopes']) == 1:
                if 'threadId' in obj['envelopes'][0]:
                    return obj['envelopes'][0]['threadId']
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': 'hound_parse',
                   'error': "tid not found in hound response"
                   }
        log_func(dumps(log_msg))
        return default
    except:
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': 'hound_parse',
                   'error': "hound parse error"
                   }
        log_func(dumps(log_msg))
        return default


def tid_by_mid(uid, mid):
    log_msg = {'unixtime': time.time(),
               'date': time.ctime(),
               'handler': mid,
               'message': "start http"
               }
    settings.http_logger.info(dumps(log_msg))
    url = '{hound_host}/filter_search?uid={uid}&mids={mid}&folder_set=default'\
          .format(hound_host=settings.conf['MessagesByThreadUri']['HoundHost'], uid=uid, mid=mid)
    req = httpclient.HTTPRequest(url, headers={'User-Agent': 'oracle_mids_mapping'},
                                 connect_timeout=1.0, request_timeout=2.0)
    http_client = httpclient.HTTPClient()
    try:
        resp = http_client.fetch(req)
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': mid,
                   'message': "got code={code}, request_time={request_time}".format(code=resp.code, request_time=resp.request_time)
                   }
        settings.http_logger.info(dumps(log_msg))
        tid = parse(resp.body, mid, settings.error_logger.info)
        return tid
    except httpclient.HTTPError as e:
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': mid,
                   'error': "proxy error, code={code}".format(code=e.code)
                   }
        settings.http_logger.error(dumps(log_msg))
        return mid
    except Exception as e:
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': mid,
                   'error': 'proxy error, reason={reason}'.format(reason=e.message)
                   }
        settings.http_logger.error(dumps(log_msg))
        return mid
    except:
        log_msg = {'unixtime': time.time(),
                   'date': time.ctime(),
                   'handler': mid,
                   'error': 'proxy error, reason=unknown'
                   }
        settings.http_logger.error(dumps(log_msg))
        return mid


class MessagesByThread(web.RequestHandler):
    @gen.coroutine
    def get(self):
        ora_mid = self.get_argument('tid')
        uid = self.get_argument('uid')
        try:
            pg_mid = fetchDBByOraMid(ora_mid, '/messages_by_thread')['pg_mid']
            pg_mid = tid_by_mid(uid, pg_mid)
        except NoMids:
            pg_mid = ora_mid
        except MultipleMids:
            raise web.HTTPError(500)
        redirect_url = '{host}:{port}/{uri}?uid={uid}&tid={pg_tid}'.format(
            uri=settings.conf['MessagesByThreadUri']['Uri'],
            host=settings.conf['MessagesByThreadUri']['RedirectHost'],
            port=settings.conf['MessagesByThreadUri']['RedirectPort'], uid=uid, pg_tid=pg_mid)

        for k, v in self.request.arguments.items():
            if k != 'tid' and k != 'uid':
                for value in v:
                    redirect_url += '&{key}={value}'.format(key=k, value=value)

        self.redirect(redirect_url)


def get_application():
    return web.Application(
        [
            (r'/ping', PingHandler),
            (r'/messages_by_thread?', MessagesByThread),
            (r'/uid_by_mid?', UidByMid),
        ]
    )
