#!/bin/bash

if [[ -v JENKINS_HOME && -v JENKINS_URL && -v JENKINS_SERVER_COOKIE ]]; then
    echo "Jenkins detected -x is set"
    set -x
fi

ROOT_DIR=${PWD}
if [[ -z $YA ]]; then
    YA=ya
fi
POLL_TIMEOUT=20

get_job_info () {
    curl -sS https://ci.yandex-team.ru/api/v1.0/checks/${1}
}

start_job () {
    YA_CHECK_ERROR=0
    YA_CHECK_RESULT=$(${YA} check 2>&1) || YA_CHECK_ERROR=$?

    if [ $YA_CHECK_ERROR -ne 0 ]; then
        echo ya check: $YA_CHECK_RESULT >&2
        exit $YA_CHECK_ERROR
    fi

    JOB_ID=$(echo ${YA_CHECK_RESULT} | grep -oP '(?<=ci.yandex-team.ru/check/).+')

    if [ "${JOB_ID}" == "" ]; then
        error "No JOB_ID found in result: ${YA_CHECK_RESULT}" 250
    fi
    echo "Job '${JOB_ID}' has been started."
}


error () {
    echo "error: $1" >&2
    exit $2
}

check_dir () {
    if [ ! -d "$1" ]; then
        error "Directory not found: $1" 255
    fi
}

check_is_number () {
    local RE='^[0-9]+$'
    if ! [[ $1 =~ $RE ]] ; then
        error "$1 is not a number" 255
    fi
}

check_is_specified () {
    if [[ -z $2 ]]; then
        error "no $1 specified" 255
    fi
}

parce_args () {
    case "$1" in
        -j|--job)
            check_is_specified "job-id" $2
            JOB_ID=$2
            shift 2
        ;;
        -p|--poll-timeout)
            check_is_specified "poll timeout" $2
            check_is_number $2
            POLL_TIMEOUT=$2
            shift 2
        ;;
        -d|--directory)
            check_is_specified "directory" $2
            ROOT_DIR=$(readlink -f $2)
            check_dir ${ROOT_DIR}
            shift 2
        ;;
        -h|--help)
            echo "Usage:"
            echo "   check_ci.sh [parameters]"
            echo
            echo "Starts CI and wait for the job completion."
            echo
            echo "Parameters:"
            echo "   -j|--job <job-id>          - attach to the existing job."
            echo "   -d|--directory <path>      - project root directory path."
            echo "   -p|--poll-timeout <path>   - poll time-out in seconds."
            echo "   -h|--help                  - show this help."
            echo
            echo "Example:"
            echo "   check_ci.sh -j a24dl -d mail/ozo"
            exit 0
        ;;
        "")
            return 0
        ;;
        *)
            error "Unknown parameter '$1' use -h for help" 255
        ;;
    esac
    parce_args $*
}

parce_args $*

cd ${ROOT_DIR}

if [ "${JOB_ID}" == "" ]; then
    start_job
fi

echo "CI url: https://ci.yandex-team.ru/check/${JOB_ID}"

set -e

JOB_STATUS="IN_PROGRESS"
PREV_PERCENTAGE=""
PREV_JOB_STATUS=""
while [ ${JOB_STATUS} != "FINISHED" ]; do
    sleep ${POLL_TIMEOUT}
    JOB_RESULT="$(get_job_info $JOB_ID)"
    JOB_STATUS="$(echo ${JOB_RESULT} | jq -r '.status')"
    PERCENTAGE=$(echo ${JOB_RESULT} | jq -r '.progress.percent')
    PERCENTAGE=$(printf %3.0f\\n $(bc <<< "${PERCENTAGE} * 100"))
    if [ "${PREV_JOB_STATUS}${PREV_PERCENTAGE}" != "${JOB_STATUS}${PERCENTAGE}" ]; then
        echo "${JOB_STATUS} ${PERCENTAGE}%"
        PREV_JOB_STATUS="${JOB_STATUS}"
        PREV_PERCENTAGE="${PERCENTAGE}"
    fi
done

CHECK_STATUS=$(echo ${JOB_RESULT} | jq -r '.progress.check_status')

echo "Check status: ${CHECK_STATUS}"

if [ "${CHECK_STATUS}" != "PASSED" ]; then
    echo ${JOB_RESULT}
    exit 200
fi

exit 0
