#!/bin/bash

if [[ -v JENKINS_HOME && -v JENKINS_URL && -v JENKINS_SERVER_COOKIE ]]; then
    echo "Jenkins detected -x is set"
    set -x
fi

set -e

if [[ -z $YA ]]; then
    YA=ya
fi

ROOT_DIR=${PWD}

error () {
    echo "error: $1" >&2
    exit $2
}

parce_args () {
    case "$1" in
        -f|--force)
            CHECKOUT_PARAMS='-f'
            shift
        ;;
        -g|--generate_ya_make)
            FORCE_GENERATE_YA_MAKE='true'
            shift
        ;;
        -d|--directory)
            if [[ -z $2 ]]; then
                error "no project directory specified" 255
            fi
            ROOT_DIR=$(readlink -f $2)
            shift 2
        ;;
        -h|--help)
            echo "Usage:"
            echo "   git_checkout.sh [parameters] {branch|commit-id}"
            echo
            echo "Checkout synchronized project's branch from git repository."
            echo
            echo "Parameters:"
            echo "   -f|--force            - discard local changes."
            echo "   -g|--generate_ya_make - force to build generate_ya_make script."
            echo "   -d|--directory <path> - project root directory path."
            echo "   -h|--help             - show this help."
            echo
            echo "Example:"
            echo "   git_checkout.sh -f -d mail/ozo origin/master"
            exit 0
        ;;
        "")
            if [[ -z $BRANCH_OR_SHA ]]; then
                error "branch or commit-id should be specified" 255
            fi
            return 0
        ;;
        -*|--*)
            error "unknown parameter '$1'" 255
        ;;
        *)
            BRANCH_OR_SHA=$1
            shift
        ;;
    esac
    parce_args $*
}

check_dir () {
    if [ ! -d "$1" ]; then
        error "directory '$1' not found" 255
    fi
}

parce_args $*

check_dir ${ROOT_DIR}

GIT_DIR="${ROOT_DIR}/github"
check_dir ${GIT_DIR}

TESTS_DIR="${ROOT_DIR}/tests"
check_dir ${TESTS_DIR}

GENERATE_YA_MAKE="${ROOT_DIR}/generate_ya_make/generate_ya_make"

cd ${GIT_DIR}

# Lazy git repository initialization
if [ ! -d ".git" ]; then
    git init
    git remote add origin $(cat .url)
    git remote update
    git checkout $CHECKOUT_PARAMS $BRANCH_OR_SHA
else
    git fetch origin
    git checkout $CHECKOUT_PARAMS $BRANCH_OR_SHA
    # In case of remote branch is tracked we should pull it
    if [ ! -z $(git status -bs | head -1 | grep -oP '(?<=\.\.\.).+') ]; then
        git pull
    fi
fi

COMMIT_ID=$(git rev-parse $BRANCH_OR_SHA)

# Update github commit id
echo $COMMIT_ID > .id

# Apply patch
PATCH="../patches/${COMMIT_ID}.patch"
if [[ -f ${PATCH} ]]; then
    echo "Applying patch ${PATCH}..."
    git apply ${PATCH}
fi

# Generate ya.make files
cd ${ROOT_DIR}

if [[ ! -f ${GENERATE_YA_MAKE} || ${FORCE_GENERATE_YA_MAKE} == 'true' ]]; then
    echo "Building generate_ya_make utility..."
    ${YA} make --checkout generate_ya_make
fi

echo "Applying generate_ya_make..."
find github/include -type f -name '*.h' | sort | ${GENERATE_YA_MAKE} > ya.make.github
cd ${TESTS_DIR}
find ../github/tests -type f -name '*.h' -o -name '*.cpp' | sort | ${GENERATE_YA_MAKE} > ya.make.github
