#pragma once

#include <ozo/connection.h>
#include <ozo/query.h>
#include <logdog/logger.h>
#include <logdog/format/tskv.h>

namespace ozo::yandex::logdog {
/**
 * @brief Wrapper to treat an object as ozo::Connection
 *
 */
template <typename Connection>
struct as_connection_attribute {
    as_connection_attribute(const Connection& conn) : conn_(conn) {}
    static_assert(ozo::Connection<Connection>, "Connection should model ozo::Connection concept");
    constexpr operator const Connection& () const noexcept { return conn_;}
    const Connection& conn_;
};

struct connection_attribute {
    static constexpr auto name = BOOST_HANA_STRING("connection");

    template <typename T>
    decltype(auto) operator = (const T& v) const {
        return ::logdog::make_attribute(*this, as_connection_attribute(v));
    }

    template <typename T>
    decltype(auto) operator = (const std::reference_wrapper<const T>& v) const {
        return ::logdog::make_attribute(*this, as_connection_attribute(v));
    }

    template <typename T>
    decltype(auto) operator = (const std::reference_wrapper<T>& v) const {
        return ::logdog::make_attribute(*this, as_connection_attribute(v.get()));
    }

    template <typename Sequence>
    constexpr decltype(auto) operator() (const Sequence& s) const {
        return get_value(s, *this);
    }
};

constexpr connection_attribute connection;

/**
 * @brief Wrapper to treat an object as ozo::Query
 *
 * @tparam Query --- type of query object
 */
template <typename Query>
struct as_query_attribute {
    as_query_attribute(const Query& query) : query_(query) {}
    static_assert(ozo::Query<Query>, "Query should model ozo::Query concept");
    constexpr operator const Query& () const noexcept { return query_;}
    const Query& query_;
};

struct query_attribute {
    static constexpr auto name = BOOST_HANA_STRING("query");

    template <typename T>
    decltype(auto) operator = (const T& v) const {
        return ::logdog::make_attribute(*this, as_query_attribute(v));
    }

    template <typename T>
    decltype(auto) operator = (const std::reference_wrapper<const T>& v) const {
        return ::logdog::make_attribute(*this, as_query_attribute(v.get()));
    }

    template <typename T>
    decltype(auto) operator = (const std::reference_wrapper<T>& v) const {
        return ::logdog::make_attribute(*this, as_query_attribute(v.get()));
    }

    template <typename Sequence>
    constexpr decltype(auto) operator() (const Sequence& s) const {
        return get_value(s, *this);
    }
};

constexpr query_attribute query;

template <typename Query, typename = boost::hana::when<true>>
struct query_has_name : std::false_type {};

template <typename Query>
struct query_has_name<Query,
    boost::hana::when_valid<decltype(get_query_name(std::declval<const Query&>()))>
> : std::true_type {};

/**
 * @brief Wrapper to treat an object as ozo::QueryName
 *
 * @tparam Query --- type of query object
 */
template <typename Query>
struct as_query_name_attribute {
    as_query_name_attribute(const Query& query) : query_(query) {}
    static_assert(ozo::Query<Query>, "Query should model ozo::Query concept");
    constexpr operator const Query& () const noexcept { return query_;}
    const Query& query_;
};

struct query_name_attribute {
    static constexpr auto name = BOOST_HANA_STRING("query.name");

    template <typename T>
    decltype(auto) operator = (const T& v) const {
        return ::logdog::make_attribute(*this, as_query_name_attribute(v));
    }

    template <typename T>
    decltype(auto) operator = (const std::reference_wrapper<const T>& v) const {
        return ::logdog::make_attribute(*this, as_query_name_attribute(v.get()));
    }

    template <typename T>
    decltype(auto) operator = (const std::reference_wrapper<T>& v) const {
        return ::logdog::make_attribute(*this, as_query_name_attribute(v.get()));
    }

    template <typename Sequence>
    constexpr decltype(auto) operator() (const Sequence& s) const {
        return get_value(s, *this);
    }
};

constexpr query_name_attribute query_name;

} // namespace ozo::yandex::logdog

namespace logdog {

template<>
struct is_key_type<::ozo::yandex::logdog::connection_attribute> : std::true_type {};

template<>
struct is_key_type<::ozo::yandex::logdog::query_attribute> : std::true_type {};

template<>
struct is_key_type<::ozo::yandex::logdog::query_name_attribute> : std::true_type {};

} // namespace logdog
