#pragma once

#include <ozo/yandex/logdog/attributes.h>
#include <ozo/connection.h>
#include <ozo/query.h>
#include <ozo/query_conf.h>
#include <logdog/format/tskv.h>
#include <yamail/data/reflection.h>

namespace ozo {
namespace yandex {

template <typename Stream, typename T>
struct stream_query_param_impl;

template <typename Stream, typename T>
inline void stream_query_param(Stream& s, const T& v) {
    stream_query_param_impl<Stream, T>::apply(s, v);
}

template <typename Stream, typename T>
struct stream_query_param_impl {
    static void apply (Stream& s, const T& v) {
        using namespace std::literals;
        if constexpr (ozo::Array<T>) {
            s << "{";
            if (!std::empty(v)) {
                stream_query_param(s, *(v.begin()));
                std::for_each(std::next(v.begin()), v.end(), [&](const auto& item) {
                    s << ",";
                    stream_query_param(s, item);
                });
            }
            s << "}";
        } else if constexpr (ozo::HanaSequence<T> || ozo::FusionSequence<T>) {
            const auto stream_param = [&s](std::string_view delim, const auto& item) {
                    s << delim;
                    stream_query_param(s, item);
                    return ","sv;
                };
            if constexpr (ozo::HanaSequence<T>) {
                boost::hana::fold(v, "("sv, stream_param);
            } else {
                boost::fusion::fold(v, "("sv, stream_param);
            }
            s << ")";
        } else {
            s << v;
        }
    }
};

template <typename Stream, typename ...Ts>
struct stream_query_param_impl<Stream, ozo::strong_typedef_wrapper<Ts...>> :
    stream_query_param_impl<Stream, typename ozo::strong_typedef_wrapper<Ts...>::base_type> {};

template <typename Stream, typename T>
struct stream_query_param_impl<Stream, std::reference_wrapper<T>> :
    stream_query_param_impl<Stream, T> {};

template <typename Stream>
struct stream_query_param_impl<Stream, std::string_view> {
    static void apply (Stream& s, std::string_view v) {
        s << '"' << v << '"';
    }
};

template <typename Stream>
struct stream_query_param_impl<Stream, std::string> :
    stream_query_param_impl<Stream, std::string_view> {
};

template <typename Stream>
struct stream_query_param_impl<Stream, const char*> :
    stream_query_param_impl<Stream, std::string_view> {
};

template <typename Stream, size_t N>
struct stream_query_param_impl<Stream, char[N]> :
    stream_query_param_impl<Stream, std::string_view> {
};

template <typename Stream, typename ... T>
struct stream_query_param_impl<Stream, std::chrono::duration<T...>> {
    static void apply (Stream& s, const std::chrono::duration<T...>& v) {
        s << '"' << yreflection::to_string(v) << '"';
    }
};

/**
 * @brief Wrapper to treat an object as ozo::QueryParams
 *
 * @tparam T --- object type
 */
template <typename T>
class query_params_as_text {
public:
    static_assert(ozo::Query<T>, "Query should model ozo::Query concept");

    constexpr query_params_as_text(const T& query) : query_(query) {}

    template <typename Stream>
    constexpr decltype(auto) to_stream(Stream&& s) const {
        using namespace std::literals;
        boost::hana::fold(ozo::get_query_params(query_), std::make_pair(1, ""sv), [&](auto ctx, const auto& p) {
            auto [idx, delim] = ctx;
            s << delim << "$" << idx << "=";
            stream_query_param(s, p);
            return std::make_pair(++idx, ","sv);
        });
        return std::forward<Stream>(s);
    }

    std::string to_string() const {
        return to_stream(std::ostringstream{}).str();
    }

private:
    const T& query_;
};

template <typename T>
inline std::ostream& operator << (std::ostream& s, const query_params_as_text<T>& v) {
    return v.to_stream(s);
}
} // namespace yandex
using namespace ::ozo::yandex;
} // namespace ozo

namespace logdog::tskv {

template <typename T>
struct to_tskv_impl<ozo::yandex::logdog::as_connection_attribute<T>> {
    template <typename Out>
    static void apply(Out& out, const char* key, const T& v) {
        const std::string name = key;
        if (!ozo::is_null_recursive(v)) {
            out << ytskv::attr(name + ".error_message", ozo::error_message(v))
                << ytskv::attr(name + ".error_context", ozo::get_error_context(v))
                << ytskv::attr(name + ".bad", ozo::connection_bad(v))
                << ytskv::attr(name + ".host", ozo::get_host(v))
                << ytskv::attr(name + ".port", ozo::get_port(v))
                << ytskv::attr(name + ".user", ozo::get_user(v))
                << ytskv::attr(name + ".database", ozo::get_database(v));
        } else {
            out << ytskv::attr(name, "null");
        }
    }
};

template <typename T>
struct to_tskv_impl<ozo::yandex::logdog::as_query_attribute<T>> {
    template <typename Out>
    static void apply(Out& out, const char* key, const T& v) {
        const std::string name = key;
        out << ytskv::attr(name + ".name", ozo::get_query_name(v))
            << ytskv::attr(name + ".text", ozo::get_query_text(v))
            << ytskv::attr(name + ".params", ozo::query_params_as_text(v));
    }
};

template <typename T>
struct to_tskv_impl<ozo::yandex::logdog::as_query_name_attribute<T>> {
    template <typename Out>
    static void apply(Out& out, const char* key, const T& v) {
        const std::string name = key;
        out << ytskv::attr(name + ".name", ozo::get_query_name(v));
    }
};

} // namespace logdog::tskv
