#pragma once

#include <ozo/connection_pool.h>
#include <ozo/failover/role_based.h>
#include <ozo/yandex/mdb/error.h>
#include <ozo/yandex/mdb/role.h>

#include <optional>
#include <string>

namespace ozo::yandex::mdb {

/**
 * @brief Simple thread-safe connection source map
 *
 * Implements functional object which returns connection source is
 * respected to a given connection string.
 *
 * @tparam ConnectionSourceCtor --- connection source constructor function
 */
template <typename ConnectionSourceCtor>
class simple_source_map {
public:
    using source_type = std::invoke_result_t<ConnectionSourceCtor, std::string>;
    using connection_type = ozo::connection_type<source_type>;
    using map_type = std::map<std::string, source_type>;

    /**
     * Construct a new simple source map object
     *
     * @param make_source --- connection source constructor
     */
    simple_source_map(ConnectionSourceCtor make_source = ConnectionSourceCtor{}, map_type initial = map_type{} )
    : sources_(std::move(initial)), make_source(make_source) {}

    /**
     * Returns connection source is respected to a given connection string.
     * If no such connection source found the connection source constructor
     * will be invoked with a given connection string as an argument.
     *
     * @param connstr --- connection string to get connection source for
     * @return source_type& --- respective source type.
     */
    source_type& operator() (const std::string& connstr) {
        const std::lock_guard<std::mutex> guard(mutex_);
        auto i = sources_.find(connstr);
        if (sources_.end() == i) {
            i = sources_.emplace(connstr, make_source(connstr)).first;
        }
        return i->second;
    }

    /**
     * Invokes given function with constant map of sources as an argument.
     * E.g., it is useful to get statistics from connection sources.
     *
     * @note The function will be invoked under internal mutex lock, so it
     * is needed to keep in mind, that the function locks all other calls to
     * the object's methods.
     *
     * @param f --- function with signature `auto(const std::map<key_type, source_type>&)`
     * @return auto --- result of `f` invocation.
     */
    template <typename Func>
    auto apply(Func&& f) const {
        const std::lock_guard<std::mutex> guard(mutex_);
        return f(sources_);
    }

private:
    mutable std::mutex mutex_;
    map_type sources_;
    ConnectionSourceCtor make_source;
};

/**
 * @brief Creates `ozo::connection_pool` object from a given connection string
 *
 * @tparam OidMap --- oid map type
 * @tparam Statistics --- statistics
 */
template <typename OidMap, typename Statistics>
class pool_from_connstr {
    OidMap oid_map_;
    Statistics statistics_;
    ozo::connection_pool_config config_;

public:
    /**
     * Construct a new pool from connstr object with attached oid map, statistics
     * and pool configuration
     *
     * @param oid_map --- oid map for connections
     * @param statistics --- statistics for connections
     * @param config --- configuration for connection pool
     */
    pool_from_connstr (OidMap oid_map, Statistics statistics, ozo::connection_pool_config config)
    : oid_map_(std::move(oid_map)), statistics_(std::move(statistics)), config_(std::move(config)) {
    }

    /**
     * Creates new instance of `ozo::connection_pool` with attached
     * oid map, statistics and settings.
     *
     * @param connstr --- connection string to be used to establish connection
     * @return `ozo::connection_pool` instance
     */
    auto operator() (const std::string& connstr) const {
        return ozo::connection_pool(connection_info(connstr, oid_map_, statistics_), config_);
    }
};

/**
 * @brief Creates simple source map of connection pools
 *
 * @param oid_map --- oid map to be used for each connection
 * @param stats --- statistics to be used for each connection
 * @param cfg --- `ozo::connection_pool` configuration
 * @return `simple_source_map` instance
 */
template <typename OidMap, typename Statistics>
inline auto make_simple_pool_map(OidMap&& oid_map, Statistics&& stats,
        ozo::connection_pool_config cfg) {
    return simple_source_map(pool_from_connstr(
        std::forward<OidMap>(oid_map), std::forward<Statistics>(stats), std::move(cfg)
    ));
}

/**
 * @brief Resolves connection string to a respective connection
 *
 * @tparam TimeConstraint --- time constaint type for connection source
 * @tparam Handler --- handler to be passed to a connection source
 * @tparam SourceMap --- connection source map functor
 */
template <typename TimeConstraint, typename Handler, typename SourceMap>
class resolve_connstr_op {
    io_context& io_;
    SourceMap sources_;
    TimeConstraint t_;
    Handler handler_;

public:
    /**
     * @brief Construct a new resolve connstr op object
     *
     * @param io --- `ozo::io_context` object reference to be used for a connection
     * @param sources --- connection source map functor
     * @param t --- time constaint type for connection source
     * @param handler --- handler to be passed to a connection source
     */
    resolve_connstr_op (io_context& io, SourceMap sources,
        TimeConstraint t, Handler handler)
    : io_(io), sources_(std::move(sources)), t_(t), handler_(std::move(handler)) {}

    /**
     * Connection type to be produced by a connection source.
     */
    using connection_type = ozo::connection_type<unwrap_type<SourceMap>>;

    /**
     * Operator should be invoked by key entity resolver (e.g.: UID-resolver, Shard-resolver)
     *
     * @param ec --- error code
     * @param conninfo --- connection string
     */
    void operator() (error_code ec, const std::string& conninfo) {
        if (ec) {
            return handler_(ec, connection_type{});
        }
        unwrap(unwrap(sources_)(conninfo))(io_, t_, std::move(handler_));
    }

    using executor_type = asio::associated_executor_t<Handler>;

    executor_type get_executor() const noexcept {
        return asio::get_associated_executor(handler_);
    }

    using allocator_type = asio::associated_allocator_t<Handler>;

    allocator_type get_allocator() const noexcept {
        return asio::get_associated_allocator(handler_);
    }
};

/**
 * @brief Resolver-based connection source for role-based failover strategy.
 *
 * This class models ConnectionSource concept for `ozo::failover::role_based_connection_provider`.
 *
 * @tparam Resolver --- resolver functional object with signature `void(Role, void(error_code, std::string))`;
 * @tparam SourceMap --- connection source map functor type
 * @tparam Role --- host role to get connection to
 */
template <typename Role, typename Resolver, typename SourceMap>
class connection_source {
    Role role_;
    Resolver resolve_;
    SourceMap source_map_;

public:
    using connection_type = ozo::connection_type<unwrap_type<SourceMap>>;
    using role_type = Role;

    /**
     * Construct a new connection source object
     *
     * @param role --- initial role for the source
     * @param resolver --- resolver functional object (e.g.: UID-resolver, Shard-resolver)
     * @param source_map --- connection source map functor
     */
    template <typename ...Ts>
    connection_source(role<Ts...> r, Resolver resolver, SourceMap source_map)
    : role_(std::move(r)),
      resolve_(std::move(resolver)),
      source_map_(std::move(source_map)) {}

    const role_type& role() const & { return role_; }
    role_type& role() & { return role_; }
    role_type&& role() && { return static_cast<role_type&&>(role_); }

    template <typename TimeConstraint, typename Handler>
    void operator() (io_context& io, TimeConstraint t, Handler&& handler) & {
        resolve_(role(), resolve_connstr_op(io, source_map_, t, std::forward<Handler>(handler)));
    }

    template <typename TimeConstraint, typename Handler>
    void operator() (io_context& io, TimeConstraint t, Handler&& handler) const & {
        resolve_(role(), resolve_connstr_op(io, source_map_, t, std::forward<Handler>(handler)));
    }

    template <typename TimeConstraint, typename Handler>
    void operator() (io_context& io, TimeConstraint t, Handler&& handler) && {
        std::move(resolve_)(std::move(role()), resolve_connstr_op(io, std::move(source_map_), t, std::forward<Handler>(handler)));
    }

    template <typename ...Ts>
    constexpr auto rebind_role(ozo::yandex::mdb::role<Ts...> r) const & {
        return connection_source<decltype(r), Resolver, SourceMap> {
            std::move(r), resolve_, source_map_
        };
    }

    template <typename ...Ts>
    constexpr auto rebind_role(ozo::yandex::mdb::role<Ts...> r) && {
        return connection_source<decltype(r), Resolver, SourceMap> {
            std::move(r), std::move(resolve_), std::move(source_map_)
        };
    }

    constexpr auto operator[] (io_context& io) const & {
        return ozo::failover::role_based_connection_provider(*this, io);
    }

    constexpr auto operator[] (io_context& io) & {
        return ozo::failover::role_based_connection_provider(*this, io);
    }

    constexpr auto operator[] (io_context& io) && {
        return ozo::failover::role_based_connection_provider(std::move(*this), io);
    }
};

template <typename Resolver, typename SourceMap, typename ...Ts>
connection_source(role<Ts...> r, Resolver resolver, SourceMap source_map) ->
    connection_source<role<Ts...>, Resolver, SourceMap>;

} // namespace ozo::yandex::mdb
