#pragma once

#include <ozo/error.h>
namespace ozo::yandex::mdb {

namespace sqlstate {

/**
 * @brief MDB-related sqlstate error conditions.
 *
 * MDB-related custom SQL states should be placed here
 */
enum code {
    ok, //!< no error placeholder
    user_removed = 37572697, //!< MDBA1 - user is not in this shard anymore, probably transferred to different shard
};

/**
 * @brief Error category for MDB-related errors
 *
 * @return reference to the category object
 */
const error_category& category() noexcept;

} // namespace sqlstate

namespace errc {

/**
 * @brief Common MDB-related error conditions
 */
enum code {
    ok, //!< no error placeholder
    user_removed, //!< user not in this shard anymore, literally same as `ozo::yandex::mdb::sqlstate::user_removed`
    database_lock_failed, //!< database lock failed
    database_read_only, //!< database in read-only mode, literally same as `ozo::errc::database_readonly`
    connection_error, //!< database communication-related error
    no_endpoint, //!< requested endpoint could not be found by resolver
    introspection_error, //<! errors related to objects serialization/deserialization, literally same as `ozo::errc::introspection_error`
    type_mismatch, //!< result type mismatch, indicates types mismatch between result of query and expected result, literally same as `ozo::errc::type_mismatch`
    protocol_error, //!< specific protocol-related errors
    too_many_connections, //!< the limit of connections for the user has been reached
};

/**
 * @brief Error conditions category
 *
 * Error conditions category object is used to construct `error_condition`.
 *
 * @return reference to the category object
 */
const error_category& category() noexcept;

} // namespace errc

} // namespace ozo::yandex::mdb


namespace boost::system {

template <>
struct is_error_condition_enum<ozo::yandex::mdb::sqlstate::code> : std::true_type {};
template <>
struct is_error_condition_enum<ozo::yandex::mdb::errc::code> : std::true_type {};

} // namespace boost::system

namespace ozo::yandex::mdb {

namespace sqlstate {

inline auto make_error_condition(code e) {
    return error_condition(static_cast<int>(e), category());
}

inline auto make_error_code(int e) {
    return error_code(e, category());
}

namespace detail {

class category final : public error_category {
public:
    const char* name() const noexcept override { return "ozo::yandex::mdb::sqlstate::category"; }

    std::string message(int value) const override {
        switch (code(value)) {
            case ok :
                return "no error";
            case user_removed :
                return "user is not in this shard anymore, probably transferred to different shard";
        };
        return "no message for value: " + std::to_string(value);
    }

    bool equivalent (const error_code& code, int condition) const noexcept override {
        if (condition == user_removed) {
            return code.category() == ozo::sqlstate::category() && code.value() == user_removed;
        }
        return error_category::equivalent(code, condition);
    }
};

} // namespace detail

inline const error_category& category() noexcept {
    const static detail::category retval;
    return retval;
}

} // namespace sqlstate

namespace errc {

inline auto make_error_condition(code e) {
    return error_condition(static_cast<int>(e), category());
}

inline auto make_error_code(int e) {
    return error_code(e, category());
}

namespace detail {

class category final : public error_category {
public:
    const char* name() const noexcept override { return "ozo::yandex::mdb::errc::category"; }

    std::string message(int value) const override {
        switch (code(value)) {
            case ok : return "no error";
            case user_removed :
                return "user not in this shard anymore, probably transferred to different shard";
            case database_lock_failed :
                return "database lock failed";
            case database_read_only :
                return "database in read-only mode";
            case connection_error :
                return "database communication-related error";
            case no_endpoint :
                return "requested endpoint could not be found by resolver";
            case introspection_error :
                return "errors related to objects serialization/deserialization";
            case type_mismatch :
                return "result type mismatch, indicates types mismatch between result of query and expected result";
            case protocol_error :
                return "specific protocol-related errors";
            case too_many_connections :
                return "the limit of connections for the user has been reached";
        };
        return "no message for value: " + std::to_string(value);
    }

    bool equivalent (const error_code& code, int condition) const noexcept override {
        const auto default_ = [&] { return error_category::equivalent(code, condition); };
        switch (ozo::yandex::mdb::errc::code(condition)) {
            case ok :
                return default_();
            case user_removed :
                return code == ozo::yandex::mdb::sqlstate::user_removed;
            case database_lock_failed :
                return code == ozo::sqlstate::lock_not_available;
            case database_read_only :
                return code == ozo::errc::database_readonly;
            case connection_error :
                return code == ozo::errc::connection_error;
            case no_endpoint :
                return default_();
            case introspection_error :
                return code == ozo::errc::introspection_error;
            case type_mismatch :
                return code == ozo::errc::type_mismatch;
            case protocol_error :
                return code == ozo::errc::protocol_error;
            case too_many_connections :
                return code == ozo::sqlstate::too_many_connections;
        }
        return default_();
    }
};

} // namespace detail

inline const error_category& category() noexcept {
    const static detail::category retval;
    return retval;
}

} // namespace errc

} // namespace ozo::yandex::mdb
