#pragma once

#include <ozo/failover/role_based.h>
#include <ozo/yandex/mdb/error.h>

#include <boost/hana/append.hpp>

#include <variant>

namespace ozo::yandex::mdb {

/**
 * @brief Error conditions constant type
 *
 * Like std::integral_constant type provides type-defined constant.
 * conditions_constant::value static member contains of the constant itself
 * - hana::tuple of errcs.
 *
 * @tparam errcs --- variadic of error conditions
 */
template <auto ...errcs>
struct conditions_constant {
    using type = conditions_constant;
    static constexpr auto value = hana::make_tuple(errcs...);
};

/**
 * @brief Role with compile-time defined traits
 *
 * @tparam Tag --- tag to identify role
 * @tparam ForceUpdate --- flag that given role forces the resolver update
 * @tparam Errcs --- conditions_constant<> with error conditions which roles should recover.
 */
template <typename Tag, typename ForceUpdate = std::false_type, typename Errcs = conditions_constant<>>
struct role : ozo::failover::role<Tag> {};

constexpr std::true_type with_force_update; //!< cache state should be updated before resolving
constexpr std::false_type no; //!< just says 'no' to things that should not be.

/**
 * @brief Indicates if given role forces the resolver update
 *
 * @param role --- role to get property of
 * @return std::true_type --- cache state should be updated before resolving
 * @return std::false_type --- current cache state may be used
 */
template <typename Tag, typename ForceUpdate, typename Errcs>
constexpr ForceUpdate force_update(const role<Tag, ForceUpdate, Errcs>&) { return {};}

/**
 * @brief Compile-time function provides role tag
 *
 * Tag should be used to identify a role
 *
 * @param role --- role to get tag of
 * @return tag type
 */
template <typename Tag, typename ForceUpdate, typename Errcs>
constexpr Tag tag(const role<Tag, ForceUpdate, Errcs>&);

/**
 * @brief Compile-time function provides role error conditions
 *
 * @param role --- role to get error conditions of
 * @return conditions_constant specialization.
 */
template <typename Tag, typename ForceUpdate, typename Errcs>
constexpr Errcs conditions(const role<Tag, ForceUpdate, Errcs>&);

template <typename Tag>
struct role_name {
    static_assert(std::is_void_v<Tag>, "No name specified for the role tag");
};

/**
 * @brief Nome of a given role dispatched by its tag
 *
 * @param v --- role object to get name of
 * @return --- boost::hana::string with role name
 */
template <typename ...Ts>
constexpr auto name(const role<Ts...>& v) {
    return role_name<decltype(tag(v))>::value;
}

/**
 * @brief Helper to construct semantic statement for fallback
 *
 * @tparam errcs --- number of error codes to recover with role
 */
template <auto ...errcs>
struct on_t {
    template <typename ...Ts, typename Forced = std::false_type>
    static constexpr auto use(const role<Ts...>& r, const Forced& = {}) {
        return role<decltype(tag(r)), Forced, conditions_constant<errcs...>>{};
    }
};

/**
 * @brief Helper to construct semantic statement for fallback
 *
 * ### Example
 *
 * Role `the_role` should recover conditions `condition1, condition2, ..., conditionN`.
 * @code
on<condition1, condition2, ..., conditionN>.use(the_role);
 * @endcode
 *
 * Role `the_role` should recover conditions `condition1, condition2, ..., conditionN`
 * and force resolver cache update if any.
 * @code
on<condition1, condition2, ..., conditionN>.use(the_role, with_force_update);
 * @endcode
 */
template <auto ...errcs>
constexpr on_t<errcs...> on;

constexpr role<class master_tag> master; //!< Master host role

template <std::size_t N> class replica_tag {
    static_assert(N>0, "replica role index should start from 1 (replica<1>)");
};

template <std::size_t N>
constexpr role<replica_tag<N>> replica; //!< Replica host role with 1-based index of replica host
constexpr role<class actual_replica_tag> actual_replica; //!< Replica host with no replication lag
constexpr role<class lagging_replica_tag> lagging_replica; //!< Replica host with replication lag

template <>
struct role_name<master_tag> {
    static constexpr auto value = BOOST_HANA_STRING("master");
};

template <>
struct role_name<actual_replica_tag> {
    static constexpr auto value = BOOST_HANA_STRING("actual_replica");
};

template <>
struct role_name<lagging_replica_tag> {
    static constexpr auto value = BOOST_HANA_STRING("lagging_replica");
};

template <std::size_t N>
struct role_name<replica_tag<N>> {
    static_assert(N<10, "replica id number greater than 9 is not supported");
    static constexpr auto value = BOOST_HANA_STRING("replica") + hana::string_c<'0' + N>;
};

} // namespace ozo::yandex::mdb

namespace ozo::failover {

template <typename ...Ts>
struct can_recover_impl<ozo::yandex::mdb::role<Ts...>> {
    static constexpr auto apply(const ozo::yandex::mdb::role<Ts...>& v, const error_code& ec) {
        return ozo::errc::match_code(decltype(conditions(v))::value, ec);
    }
};

} // namespace ozo::failover
