#ifndef _PA_PROFILER_ASYNC_H_
#define _PA_PROFILER_ASYNC_H_

#include <deque>
#include <vector>
#include <string>
#include <sys/uio.h>
#include <pa/logger.h>
#include <pa/singleton.h>
#include <pa/interface.h>
#include <boost/thread/thread.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition_variable.hpp>
#include <boost/thread/locks.hpp>

#include <boost/chrono/duration.hpp>

#define PA_PROFILER_ASYNC_SINGL singleton< std::unique_ptr<impl> >::get()

namespace pa {

class async_profiler
{
public:
  enum {
    OPT_SAVE_QUEUE = 0x01,
    OPT_SAVE_OLD   = 0x02
  };
private:
  typedef boost::mutex mutex_t;
  typedef boost::unique_lock<mutex_t> lock_t;
  typedef std::deque<wmi_profiler_entry*> queue_t;

  class impl {
  public:
    impl(std::size_t ms, std::size_t ds, const char* f, unsigned opt)
      : stopped (false)
      , max_size (ms)
      , dump_size (ds)
      , log_ (f)
      , options (opt)
    {}

    impl(std::size_t ms, std::size_t ds, const std::string& f, unsigned opt)
      : stopped (false)
      , max_size (ms)
      , dump_size (ds)
      , log_ (f)
      , options (opt)
    {}

    void start()
    {
      io_thread.reset(new boost::thread(&impl::io_thread_f, this));
    }

    ~impl()
    {
      if (io_thread.get()) {
        {
          lock_t lock(mux);
          stopped = true;
        }
        io_thread->join();
      }
    }

    inline void add_to_queue(rem_type type, const string &host, const string &req, const string &suid, uint32_t spent_ms, time_t tm)
    {
      wmi_profiler_entry* new_data = new wmi_profiler_entry(type, host, req, suid, spent_ms, tm);
      {
        lock_t lock(mux);
        if (stopped) { delete new_data; return; }
        if (options & OPT_SAVE_OLD) {
          if (queue_.size() >= max_size)
            delete new_data;
          else
            queue_.push_back(new_data);
        } else {
          queue_.push_back(new_data);
          if (queue_.size() > max_size) {
            new_data = queue_.front();
            queue_.pop_front();
            delete new_data;
          }
        }
        if (queue_.size() > dump_size) return;
      }
      cond.notify_one();
    }

  private:
    void io_thread_f()
    {
      while (true) {
        try { if (!do_step()) break; } catch (...) {}
      }
    }

    bool do_step()
    {
      queue_t tmp;
      {
        lock_t lock(mux);
        while (queue_.empty() && !stopped)
          cond.wait_for(lock, boost::chrono::milliseconds(100));
        if (stopped && queue_.empty()) return false;
        tmp.swap(queue_);
      }
      if (log_.open ()) {
        log_.write (tmp.begin (), tmp.end ());
        std::for_each (tmp.begin (), tmp.end (), boost::bind (&::operator delete, _1));
        log_.close ();
      } else if (options & OPT_SAVE_QUEUE) {
        lock_t lock(mux);
        for (queue_t::reverse_iterator i = tmp.rbegin(), i_end = tmp.rend(); i != i_end; ++i) {
          if (queue_.size() > max_size) {
            if (options & OPT_SAVE_OLD) break;
            else {
              delete *queue_.rbegin();
              queue_.pop_back();
            }
          } else {
            queue_.push_front(*i);
          }
        }
      }
      return true;
    }

    bool stopped;
    std::size_t max_size;
    std::size_t dump_size;
    pa::logger<wmi_profiler_entry> log_;
    unsigned options;
    queue_t queue_;
    std::unique_ptr<boost::thread> io_thread;
    boost::mutex mux;
    boost::condition_variable cond;
  };

public:
  static void init(std::size_t max_size, std::size_t dump_size,
                   const char* file = PROFILER_LOG_PATH, unsigned options = OPT_SAVE_QUEUE)
  {
    PA_PROFILER_ASYNC_SINGL.reset(new impl(max_size, dump_size, file, options));
    PA_PROFILER_ASYNC_SINGL->start();
  }

  static void init(std::size_t max_size, std::size_t dump_size,
                   const std::string& file, unsigned options = OPT_SAVE_QUEUE)
  {
    PA_PROFILER_ASYNC_SINGL.reset(new impl(max_size, dump_size, file, options));
    PA_PROFILER_ASYNC_SINGL->start();
  }

  static bool is_init() {
    return PA_PROFILER_ASYNC_SINGL.get() != nullptr;
  }

  void restart_timer() {
    tmr.start();
  }

  inline void add_internal_time(rem_type type, const string &host, const string &req, const string &suid, time_t tm = time(NULL))
  {
    add(type, host, req, suid, tmr.stop(), tm);
    restart_timer();
  }

  static void add(rem_type type, const string &host, const string &req, const string &suid, uint32_t spent_ms, time_t tm = time(NULL))
  {
    if (!PA_PROFILER_ASYNC_SINGL.get()) return;
    try {
      PA_PROFILER_ASYNC_SINGL->add_to_queue(type, host, req, suid, spent_ms, tm);
    } catch (...) {}
  }
private:
  stimer_t tmr;
};

}

#undef PA_PROFILER_ASYNC_SINGL
#endif // _PA_PROFILER_ASYNC_H_
