#ifndef _PA_PROFILER_H_
#define _PA_PROFILER_H_

#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <string>
#include <pa/logger.h>
#include <pa/stimer.h>

#ifndef _FILE_OFFSET_BITS
#define _FILE_OFFSET_BITS 64
#endif

#ifndef _LARGEFILE_SOURCE
#define _LARGEFILE_SOURCE
#endif

#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif

#ifndef PROFILER_LOG_PATH
#define PROFILER_LOG_PATH "/var/log/profiler.log"
#endif

namespace pa {

using std::string;

/*
 * New types must be defined only in PROFILER_REM_TYPES definition  block
 */
#define PROFILER_REM_TYPES(DEFINE) \
    DEFINE(mysql) \
    DEFINE(oracle) \
    DEFINE(mulca) \
    DEFINE(antivirus) \
    DEFINE(spam) \
    DEFINE(smtp_out) \
    DEFINE(passport) \
    DEFINE(transformers) \
    DEFINE(aproxy) \
    DEFINE(rz) \
    DEFINE(js) \
    DEFINE(smtp_client) \
    DEFINE(tormoz) \
    DEFINE(search) \
    DEFINE(etc) \
    DEFINE(http_getter) \
    DEFINE(wmi_method) \
    DEFINE(mongo) \
    DEFINE(memcached) \
    DEFINE(hbase) \
    DEFINE(message_body) \
    DEFINE(mailbox_oper) \
    DEFINE(zookeeper) \
    DEFINE(postgresql) \
    DEFINE(xiva) \
    DEFINE(furita) \
    DEFINE(postmaster) \
    DEFINE(docviewer) \
    DEFINE(settings) \
    DEFINE(fastsrv) \
    DEFINE(sharpei) \
    DEFINE(xeno) \
    DEFINE(doberman) \
    DEFINE(imap_client) \
    DEFINE(_last_type)

#define PROFILER_ENUMERATOR(TYPE) TYPE,
    typedef enum {
        unknown=0,
        PROFILER_REM_TYPES(PROFILER_ENUMERATOR)
    } rem_type;

inline string make_mulca_host (const string &mid) {
  string::size_type dot = mid.find('.');
  return dot == string::npos ? mid : mid.substr(0, dot);
}

struct wmi_profiler_entry {
  wmi_profiler_entry() {}
  wmi_profiler_entry(rem_type type_u, const string &host_u, const string &req_u, const string &suid_u, uint32_t spent_ms_u, time_t tm)
    : type(type_u), spent_ms(spent_ms_u), timestamp(static_cast<uint32_t>(tm))
  {
    strncpy(host, host_u.c_str(), sizeof(host));
    strncpy(req, req_u.c_str(), sizeof(req));
    strncpy(suid, suid_u.c_str(), sizeof(suid));
  }
  
  bool is_valid () const
  { return type <= _last_type; }

  uint32_t type;
  char host[16];
  char req[24];
  char suid[16];
  uint32_t spent_ms;
  uint32_t timestamp;
};

class wmi_profiler
{
public:
  string mulca_host(const string &mid) {
    return make_mulca_host (mid);
  }
  void restart_timer() {
    tmr.start();
  }

  static void add(rem_type type, const string &host, const string &req, const string &suid, uint32_t spent_ms, time_t tm = time(NULL), const char *profiler_log = PROFILER_LOG_PATH) {
    wmi_profiler_entry new_data(type, host, req, suid, spent_ms, tm);
    pa::logger<wmi_profiler_entry> log (profiler_log);
    if (log.open ()) {
      log.write (&new_data);
    }
  }

  inline void add(rem_type type, const string &host, const string &req, const string &suid, time_t tm = time(NULL), const char *profiler_log = PROFILER_LOG_PATH) {
    add(type, host, req, suid, tmr.stop(), tm, profiler_log);
    restart_timer();
  }

private:
  stimer_t tmr;
};

}

#endif // _PA_PROFILER_H_
