#ifndef _PA_PROFILER_LOGGER_H_
#define _PA_PROFILER_LOGGER_H_

#include <string>
#include <vector>
#include <cstring>
#include <fcntl.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <boost/bind.hpp>
#include <boost/thread/once.hpp>

#ifdef __APPLE__
#define O_LARGEFILE 0
#endif

#define PA_IGNORE_RESULT(expression) \
    do { \
        const auto __expression_result_13_42 = (expression); \
        (void) __expression_result_13_42; \
    } while (false)

namespace pa {

template <typename Entry>
class logger
{
public:
  logger (const std::string& fname)
    : fname_ (fname), fd_ (-1)
  {
    static boost::once_flag flag = BOOST_ONCE_INIT;
    boost::call_once (flag, boost::bind (&logger::do_open, this));
  }

  ~logger ()
  {
    close ();
  }

  void write (const Entry* entry)
  {
    assert (fd_ >= 0);
    PA_IGNORE_RESULT(::write (fd_, entry, sizeof (Entry)));
  }
  template <typename Iter>
  void write (Iter begin, Iter end)
  {
    assert (fd_ >= 0);
    while (begin != end) {
      PA_IGNORE_RESULT(::write (fd_, *begin, sizeof (Entry)));
      ++begin;
    }
  }
  template <typename Iter>
  bool writev (Iter begin, Iter end)
  {
    assert (fd_ >= 0);
    std::vector<iovec> buff (std::distance (begin, end));
    int i_buff = 0;
    for (Iter i = begin; i != end; ++i, ++i_buff) {
      buff[i_buff].iov_base = *i;
      buff[i_buff].iov_len = sizeof(**i);
    }
    return (::writev (fd_, &buff[0], i_buff) > 0);
  }

  void read (Entry* entry)
  {
    assert (fd_ >= 0);
    PA_IGNORE_RESULT(::read (fd_, entry, sizeof (Entry)));
  }

  bool open ()
  {
    if (fd_ >= 0) return true;
    fd_ = ::open (fname_.c_str(), O_CREAT|O_WRONLY|O_APPEND|O_LARGEFILE,0664);
    return fd_ >= 0;
  }
  void close ()
  {
    if (fd_ < 0) return;
    ::close(fd_);
    fd_ = -1;
  }
private:
  void do_open ()
  {
    fd_ = ::open (fname_.c_str(), O_CREAT|O_RDWR|O_LARGEFILE, 0664);
    if (fd_ < 0) return;
    struct stat sb;
    int status = ::fstat(fd_,&sb);
    assert(status >= 0);
    (void) status;
    auto correct_pos = sb.st_size;
    if((sb.st_size % static_cast<off_t>(sizeof(Entry))) != 0) {
      correct_pos = sb.st_size / static_cast<off_t>(sizeof(Entry));
      correct_pos *= sizeof (Entry);
      ::lseek (fd_, correct_pos, SEEK_SET);
    }

    bool foundValidEntry = false;
    while (correct_pos > 0 && !foundValidEntry) {
        correct_pos -= sizeof(Entry);
        PA_IGNORE_RESULT(::lseek(fd_, correct_pos, SEEK_SET));
        Entry entry;
        PA_IGNORE_RESULT(::read(fd_, &entry, sizeof(Entry)));
        foundValidEntry = entry.is_valid();
    }
    if (foundValidEntry) {
        correct_pos += sizeof(Entry);
    }

    PA_IGNORE_RESULT(::ftruncate (fd_, correct_pos));
    close ();
  }

  std::string fname_;
  int fd_;
};

}

#endif // _PA_PROFILER_LOGGER_H_
