#ifndef _PA_TIME_HELPERS_H_
#define	_PA_TIME_HELPERS_H_

#include <ostream>
#include <time.h>

namespace pa { namespace detail {

template <typename T>
long int& fetch_division (T& v);

inline long int& fetch_division (timeval& v)
{ return v.tv_usec; }
inline long int& fetch_division (timespec& v)
{ return v.tv_nsec; }
inline const long int& fetch_division (const timeval& v)
{ return v.tv_usec; }
inline const long int& fetch_division (const timespec& v)
{ return v.tv_nsec; }

template <typename T, int D>
T sub (const T& lhs, const T& rhs)
{
  T result;
  result.tv_sec = lhs.tv_sec - rhs.tv_sec;
  fetch_division(result) = fetch_division(lhs) - fetch_division(rhs);
  if (fetch_division(lhs) < fetch_division(rhs)) {
    --result.tv_sec;
    fetch_division(result) += D;
  }
  return result;
}

template <typename T, int D>
T add (const T& lhs, const T& rhs)
{
  T result;
  result.tv_sec = lhs.tv_sec + rhs.tv_sec;
  fetch_division(result) = fetch_division(lhs) + fetch_division(rhs);
  if (fetch_division(result) >= D) {
    ++result.tv_sec;
    fetch_division(result) -= D;
  }
  return result;
}

template <typename T, int D>
T& assign (T& lhs, const T& rhs)
{
  lhs.tv_sec = rhs.tv_sec;
  fetch_division(lhs) = fetch_division(rhs);
  return lhs;
}

template <typename T>
bool cmp (const T& lhs, const T& rhs)
{
  if (lhs.tv_sec < rhs.tv_sec) return true;
  if (lhs.tv_sec > rhs.tv_sec) return false;
  if (fetch_division(lhs) < fetch_division(rhs)) return true;
  return false;
}

}
// ********** timespec
inline timespec operator- (const timespec& lhs, const timespec& rhs)
{ return detail::sub<timespec, 1000000000> (lhs, rhs); }

inline timespec operator+  (const timespec& lhs, const timespec& rhs)
{ return detail::add<timespec, 1000000000>(lhs, rhs); }

inline timespec& operator+= (timespec& lhs, const timespec& rhs)
{ return detail::assign<timespec, 1000000000> (lhs, lhs + rhs); }

inline timespec& operator-= (timespec& lhs, const timespec& rhs)
{ return detail::assign<timespec, 1000000000> (lhs, lhs - rhs); }

inline bool operator== (const timespec& lhs, const timespec& rhs)
{ return lhs.tv_sec == rhs.tv_sec && lhs.tv_nsec == rhs.tv_nsec; }

inline bool operator!= (const timespec& lhs, const timespec& rhs)
{ return !(lhs == rhs); }

inline bool operator<  (const timespec& lhs, const timespec& rhs)
{ return detail::cmp <timespec> (lhs, rhs); }

inline bool operator<= (const timespec& lhs, const timespec& rhs)
{ return (lhs < rhs || lhs == rhs); }

inline bool operator>  (const timespec& lhs, const timespec& rhs)
{ return !(lhs <= rhs); }

inline bool operator>= (const timespec& lhs, const timespec& rhs)
{ return (lhs > rhs || lhs == rhs); }

// ********** timeval
inline timeval operator- (const timeval& lhs, const timeval& rhs)
{ return detail::sub<timeval, 1000000> (lhs, rhs); }

inline timeval operator+  (const timeval& lhs, const timeval& rhs)
{ return detail::add<timeval, 1000000>(lhs, rhs); }

inline timeval& operator+= (timeval& lhs, const timeval& rhs)
{ return detail::assign<timeval, 1000000> (lhs, lhs + rhs); }

inline timeval& operator-= (timeval& lhs, const timeval& rhs)
{ return detail::assign<timeval, 1000000> (lhs, lhs - rhs); }

inline bool operator== (const timeval& lhs, const timeval& rhs)
{ return lhs.tv_sec == rhs.tv_sec && lhs.tv_usec == rhs.tv_usec; }

inline bool operator!= (const timeval& lhs, const timeval& rhs)
{ return !(lhs == rhs); }

inline bool operator<  (const timeval& lhs, const timeval& rhs)
{ return detail::cmp <timeval> (lhs, rhs); }

inline bool operator<= (const timeval& lhs, const timeval& rhs)
{ return (lhs < rhs || lhs == rhs); }

inline bool operator>  (const timeval& lhs, const timeval& rhs)
{ return !(lhs <= rhs); }

inline bool operator>= (const timeval& lhs, const timeval& rhs)
{ return (lhs > rhs || lhs == rhs); }

// ********** convertors
inline int timespec2milliseconds(const timespec& val)
{ return static_cast<int>(val.tv_sec * 1000 + val.tv_nsec / 1000000); }

inline int timeval2milliseconds(const timeval& val)
{ return static_cast<int>(val.tv_sec * 1000 + val.tv_usec / 1000); }

// for back-compatible only
inline timeval& assign_timeval (timeval& lhs, const timeval& rhs)
{ return detail::assign<timeval, 1000000> (lhs, rhs); }

}

#endif	// _PA_TIME_HELPERS_H_
