#include <iostream>
#include <string>
#include <map>
#include <boost/tokenizer.hpp>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <fstream>
#include <sstream>
#include <errno.h>
#include <cstring>
#include <cstdlib>
#include <unordered_map>

#include <pa/interface.h>

using namespace std;
using namespace pa;

namespace {

off_t lseek_checked(int fd, off_t offset, int whence) {
    off_t pos = lseek(fd, offset, whence);
    if ((off_t)-1 == pos) {
        cerr << "lseek to offset " << offset << "falied: " << strerror(errno) << endl;
        exit(EXIT_FAILURE);
    }

    return pos;
}

void fstat_checked(int fd, struct stat* statbuf) {
    if (-1 == fstat(fd, statbuf)) {
        cerr << "fstat falied: " << strerror(errno) << endl;
        exit(EXIT_FAILURE);
    }
}

} // namespace

const off_t wpe_size = static_cast<off_t>(sizeof(wmi_profiler_entry));

// map rem_type -> acceptable maximum of duration (ms)
// acceptable maximum of zero means "not specified"
std::unordered_map<uint32_t, long long> anomaly_map;
void initialize_anomaly_map() {
    anomaly_map[unknown] = 300;
    anomaly_map[mysql] = 500;
    anomaly_map[oracle] = 1000;
    anomaly_map[mulca] = 2000;
    anomaly_map[antivirus] = 500;
    anomaly_map[spam] = 500;
    anomaly_map[smtp_out] = 500;
    anomaly_map[passport] = 200;
    anomaly_map[transformers] = 200;
    anomaly_map[aproxy] = 500;
    anomaly_map[rz] = 200;
    anomaly_map[js] = 300;
    anomaly_map[smtp_client] = 2000;
    anomaly_map[mongo] = 100;
    anomaly_map[memcached] = 10;
}

#define PROFILER_STRINGIFICATOR(TYPE) #TYPE,
const char* service_name[] = {
    "unknown",
    PROFILER_REM_TYPES(PROFILER_STRINGIFICATOR)
};

const size_t num_services = sizeof(service_name)/sizeof(service_name[0]);

string fit(const char* s, size_t slen, size_t length) {
    if(slen == static_cast<size_t>(-1)) slen = strlen(s);
    return string(s, slen > length ? length : slen);
}

string convert_time(time_t ts) {
    struct tm *t = localtime(&ts);
    char buf[10];
    sprintf(buf, "%02d:%02d:%02d", t->tm_hour, t->tm_min, t->tm_sec);
    return buf;
}

void handle_entry(wmi_profiler_entry *e) {
    if (!e->is_valid ()) return;
    size_t type = e->type;
    size_t service_name_index = type < num_services ? type : 0;
    const char *t = service_name[service_name_index];
    char buf[512];
    auto len = snprintf(buf, 512, "%s   %-12s %-16s %-24s %-16s %.3f\n",
            convert_time(e->timestamp).c_str(),
            fit(t, static_cast<size_t>(-1), 12).c_str(),
            fit(e->host, 16, 16).c_str(),
            fit(e->req, 24, 24).c_str(),
            fit(e->suid, 16, 16).c_str(),
            (e->spent_ms/1000.0));
    PA_IGNORE_RESULT(write(1, buf, static_cast<std::size_t>(len)));
}

struct pa_flags
{
    bool tail_mode;
    int time_diff;
    int min_ms;
    long long services;
    unsigned int count;
    string profiler_log;
    bool verbose;
    bool show_only_count;
    bool anomaly_check;

    pa_flags():
        tail_mode(false),
        time_diff(-1),
        min_ms(-1),
        services(4294967295),
        count(0),
        profiler_log(PROFILER_LOG_PATH),
        verbose(false),
        show_only_count(false),
        anomaly_check(false)
    {
    }

};

class entry_filter
{
public:
    pa_flags flags;
    time_t min_time;

    entry_filter(pa_flags _flags) : flags(_flags)
    {
        if (flags.time_diff < 0)
            min_time = -1;
        else
            min_time = time(0) - flags.time_diff;
    }

    bool operator()(wmi_profiler_entry* e)
    {
        if (static_cast<long long>(e->spent_ms) < static_cast<long long>(flags.min_ms))
            return false;
        if (static_cast<long long>(e->timestamp) < static_cast<long long>(min_time))
            return false;
        if (flags.anomaly_check && (!anomaly_map[e->type] ||
                static_cast<long long>(e->spent_ms) < anomaly_map[e->type]))
            return false;
        return flags.services & (1<<e->type);
    }

    bool is_trivial()
    {
        return flags.min_ms < 0 && min_time < 0 && flags.services == 4294967295;
    }
};

void read_at(int fd, off_t off, wmi_profiler_entry* e)
{
    lseek_checked(fd,off,SEEK_SET);
    PA_IGNORE_RESULT(read(fd,e,wpe_size));
}

off_t find_time_point(int fd, time_t t)
{
    struct stat sb;
    fstat_checked(fd,&sb);
    if(sb.st_size%wpe_size) {
    cerr << "Strange length of log, seems to be incorrect format" << endl;
    exit(1);
    }
    off_t start_off = 0;
    off_t end_off = sb.st_size;

    wmi_profiler_entry l;
    wmi_profiler_entry r;
    read_at(fd, start_off, &l);
    read_at(fd, end_off-wpe_size, &r);
    long long time = t;
    if (static_cast<long long>(l.timestamp) >= time)
        return start_off;
    if (static_cast<long long>(r.timestamp) < time)
        return end_off;

    while (end_off - start_off > wpe_size)
    {
        off_t ln = start_off/wpe_size;
        off_t rn = end_off/wpe_size;
        off_t estimated_n = (ln + rn)/2;

        wmi_profiler_entry m;
        read_at(fd, estimated_n*wpe_size, &m);
        if (static_cast<long long>(m.timestamp) >= time)
        {
            end_off = estimated_n*wpe_size;
            r.timestamp = m.timestamp;
        } else {
            start_off = estimated_n*wpe_size;
            l.timestamp = m.timestamp;
        }
    }
    return end_off;
}

void print_help()
{
    cout << "Profiler analyser." << endl;
    cout << "Usage: pa [-l] [-f] [-F max_spent_ms] [-t timestamp_diff] [-c count] [-s services_mask] [-S service1name+service2name+serviceN] [logfile]" << endl;
    cout << "\t-f - tail mode flag. Display last 15 records and wait for input" << endl;
    cout << "\t-F - filter records by spent_ms" << endl;
    cout << "\t-a - show records with anomaly long timings only" << endl;
    cout << "\t-t - show records not older then timestamp_diff seconds" << endl;
    cout << "\t-c - display last count records" << endl;
    cout << "\t-s - filter by service code" << endl;
    cout << "\tservice codes:" << endl;
    for (size_t r=0; r+1 < num_services; r++)
        cout << "\t\t" << (1<<r) << " : " << service_name[r] << endl;
    cout << "\t-S - filter by service names" << endl;
    cout << "\t-l output only number of records filtered" << endl;
}

int build_smask_from_names(string services)
{
    int result = 0;
    map<string,int> smap;
        for (size_t r=0; r < num_services; r++)
        smap[service_name[r]] =  (1<<r);
    smap["all"] = (1<<16) - 1;

    boost::char_separator<char> sep("", "+-");
    typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
    tokenizer tokens(services, sep);

    bool operator_add = true;
    for (tokenizer::iterator tok_iter = tokens.begin(); tok_iter != tokens.end(); ++tok_iter)
    {
        if (*tok_iter == "-")
        {
            operator_add = false;
            continue;
        }
        if (*tok_iter == "+")
        {
            operator_add = true;
            continue;
        }
        int bit = smap[*tok_iter];
        if (operator_add)
            result |= bit;
        else
            result &= ~bit;
    }

    return result;
}

ostream& operator<<(ostream& os, const pa_flags& f)
{
    os << "time diff: " << f.time_diff << " (current time is " <<  time(0) << "),  tail mode:" << (f.tail_mode ? "on" : "off") << ", records count: " << f.count << ", log file:" << f.profiler_log << endl;
    os << "services to show:" << endl;
    for (size_t serv_type=0; serv_type < num_services; ++serv_type)
    {
        if (f.services & (1<<serv_type) )
        {
            os << "+";
        } else 
            os << "-";
        os << service_name[serv_type] << endl;
    }
    return os;
}

int init_from_arglist(vector<string> argv, pa_flags& flags)
{
    for(vector<string>::iterator c = argv.begin(); c<argv.end(); ++c) 
    {
        if (c->empty())
            continue;

        if((*c)[0]=='-')
        {
            if (c->size() == 1)
                continue;

            switch((*c)[1]) {
                case 'f':
                    flags.tail_mode=true;
                    flags.count = 15;
                    break;
                case 'a':
                    flags.anomaly_check = true;
                    break;
                case 'F':
                    if (c+1 != argv.end())
                    {
                        ++c;
                        flags.min_ms = static_cast<int>(atof(c->c_str()))*1000;
                    } else {
                        cerr << "No argument for -F, please use -F time_in_ms\n";
                        return 1;
                    }
                    break;
                case 't':
                    if (c+1 != argv.end())
                    {
                        ++c;
                        flags.time_diff = atoi(c->c_str());
                    } else {
                        cerr << "No argument for -t, please use -t time_diff\n";
                        return 1;
                    }
                    break;
                case 's':
                    if (c+1 != argv.end())
                    {
                        ++c;
                        flags.services = atoi(c->c_str());
                    } else {
                        cerr << "No argument for -s, please use -s service_code\n";
                        return 1;
                    }
                    break;
                case 'S':
                    if (c+1 != argv.end())
                    {
                        ++c;
                        flags.services = build_smask_from_names(c->c_str());
                    } else {
                        cerr << "No argument for -S, please use -S service1_name+service2_name\n";
                        return 1;
                    }
                    break;
                case 'c':
                    if (c+1 != argv.end())
                    {
                        ++c;
                        flags.count = static_cast<unsigned>(atoi(c->c_str()));
                    } else {
                        cerr << "No argument for -c, please use -c count";
                        return 1;
                    }
                    break;
                case 'h':
                    print_help();
                    exit(0);
                    break;
                case 'v':
                    flags.verbose = true;
                    break;
                case 'l':
                    flags.show_only_count = true;
                    break;
                case '-':
                    string long_opt = c->substr(2);
                    if (long_opt == "help")
                    {
                        print_help();
                        exit(0);
                    }
                    if (long_opt == "verbose")
                    {
                        flags.verbose = true;
                        break;
                    }
                    break;
                }
        } else {
            flags.profiler_log=*c;
        }
    }
    return 0;
}

int init_from_argstring(string arg, pa_flags& flags)
{
    boost::escaped_list_separator<char> sep("\\", " \t\r\n", "\"\"");
    boost::tokenizer<boost::escaped_list_separator<char> > tokens(arg, sep);
    vector<string> arglist;
    arglist.insert(arglist.end(), tokens.begin(), tokens.end());
    return init_from_arglist(arglist, flags);
}

int init_from_rc(pa_flags& flags)
{
    string alltext;

    stringstream rc_name;
    rc_name << getenv ("HOME") << "/.parc";
    ifstream parc(rc_name.str().c_str(), ifstream::in);
    while (parc.good())
    {
        char buff[256];
        parc.getline(buff, 256);
        alltext+= " ";
        alltext+= buff;
    }
    return init_from_argstring(alltext, flags);
}

int main(int argc, char **argv) 
{
    initialize_anomaly_map();

    pa_flags flags; 

    if (init_from_rc(flags))
        return 1;

    char *env_args = getenv("PA_RC");
    if (env_args)
    {
        if (init_from_argstring(env_args, flags))
            return 1;
    }

    vector<string> arglist;
    arglist.insert(arglist.end(), argv + 1, argv + argc);
    if (init_from_arglist(arglist, flags))
        return 1;

    if (flags.verbose)
        cout << flags;

    int fd=open(flags.profiler_log.c_str(),O_RDONLY|O_LARGEFILE);
    if(fd<1) {
        cerr << "Can't open " << flags.profiler_log << endl;
        return 1;
    }

    //check file size is multiple of wpe_size
    struct stat sb;
    fstat_checked(fd,&sb);

    if (flags.count > 0) //rewind back count records satisfying filter
    {
        //reset time_diff, -c has higher priority than -t
        flags.time_diff = -1;
    }

    entry_filter filt(flags);

    if (flags.count > 0) //rewind back count records
    {
        if (sb.st_size/wpe_size > flags.count)
        {
            off_t newoff=sb.st_size-(flags.count*wpe_size);
            lseek_checked(fd,newoff,SEEK_SET);
        } else { //put pointer at the beginnig of file
            lseek_checked(fd,0,SEEK_SET);
        }

    } else if (filt.min_time > 0) //try to set current position close to starting time
    {
        off_t newoff = find_time_point(fd, filt.min_time);
        lseek_checked(fd,newoff,SEEK_SET);
    }

    int num_records_passed = 0;

    char buf[wpe_size];
    off_t already=0;
    while(true)
    {
        ssize_t r=read(fd,buf+already,static_cast<std::size_t>(wpe_size-already));
        if((already+r) != wpe_size)
        {
            if(r==0) {
                if(!flags.tail_mode)
                    break;
                usleep(250000);
                continue;
            }
            if(r<0) {
                cerr << "Can't read from " << flags.profiler_log << ": " << strerror(errno) << endl;
                return 1;
            }
            already+=r;
            continue;
        }
        wmi_profiler_entry *e = reinterpret_cast<wmi_profiler_entry*>(buf);
        if (filt(e))
        {
            if (!flags.show_only_count)
                handle_entry(e);
            num_records_passed++;
        }

        //check that we are not passed original size (output only records existing when pa was launched)
        off_t cur = lseek_checked(fd, 0, SEEK_CUR);
        if (cur >= sb.st_size && !flags.tail_mode)
            break;

        already=0;
    }
    if (flags.show_only_count)
        cout << num_records_passed << endl;
    return 0;
}
