package api

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"context"
	"fmt"
	"net/url"
	"os"
	"strconv"
	"strings"
)

func validateOAuthAndUIDPair(ctx context.Context, bb blackbox.Client, uid string, token string, userIP string) (context.Context, error) {
	logger := ctxutil.GetLogger(ctx)
	reqID := ctxutil.GetRequestID(ctx)

	if len(token) == 0 {
		logger.Info(fmt.Sprintf("Unauthorized request incoming [%s].", reqID), ctxutil.GetStoredFields(ctx)...)
		return ctx, nil
	}
	if !strings.HasPrefix(token, OAuthTokenPrefix) {
		logger.Error("Incorrect Authorization header format",
			append(ctxutil.GetStoredFields(ctx), log.String(AuthHeaderKeyXUid, uid))...)
		return ctx, fmt.Errorf("authorization failed: improper Authorization token")
	}

	bbResponse, err := bb.OAuth(ctx, blackbox.OAuthRequest{
		OAuthToken: token[len(OAuthTokenPrefix):],
		UserIP:     userIP,
		GetLoginID: true,
	})
	if err != nil {
		logger.Error(fmt.Sprintf("%v", err))
		return ctx, fmt.Errorf("authorization failed: bad OAuth token value")
	}

	if len(bbResponse.LoginID) > 0 {
		ctx = ctxutil.WithLoginID(ctx, bbResponse.LoginID)
	}

	if strconv.FormatUint(bbResponse.User.ID, 10) != uid {
		logger.Error("Validating OAuth and uid pair failed",
			append(ctxutil.GetStoredFields(ctx), log.String(AuthHeaderKeyXUid, uid))...)
		return ctx, fmt.Errorf("authorization failed: nonexistent OAuth token")
	}
	// TODO: place this in config
	scopeToValidate, scopeNameFound := os.LookupEnv("SCOPE_TO_VALIDATE")
	if !scopeNameFound {
		return ctx, nil
	}
	scopeToValidateDecoded, err := url.QueryUnescape(scopeToValidate)
	if err != nil {
		logger.Errorf("could not url-decode env variable SCOPE_TO_VALIDATE: %v", err)
		return ctx, fmt.Errorf("authorization failed: OAuth token did not passed scope check")
	}
	for _, currentScope := range bbResponse.Scopes {
		logger.Info(
			fmt.Sprintf("checking scope %s againts allowed scope %s", currentScope, scopeToValidateDecoded),
			append(ctxutil.GetStoredFields(ctx), log.String(AuthHeaderKeyXUid, uid))...,
		)
		if currentScope == scopeToValidateDecoded {
			return ctx, nil
		}
	}
	logger.Error(
		fmt.Sprintf("scope %s is not in the allowed scopes list for user", scopeToValidateDecoded),
		append(ctxutil.GetStoredFields(ctx), log.String(AuthHeaderKeyXUid, uid))...,
	)
	return ctx, fmt.Errorf("authorization failed: OAuth Scope validation failed")
}
