package httputil

import (
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"encoding/json"
	"fmt"
	"golang.org/x/net/context"
	"net/http"
)

type ServiceErrorStatusCode int

const (
	ServiceErrorStatusCodeIncorrectFormat     ServiceErrorStatusCode = 1001
	ServiceErrorStatusCodeProxyCall           ServiceErrorStatusCode = 1002
	ServiceErrorStatusCodeInfra               ServiceErrorStatusCode = 1003
	ServiceErrorStatusCodeAuth                ServiceErrorStatusCode = 1004
	ServiceErrorStatusCodeMissingServiceToken ServiceErrorStatusCode = 1005
)

type ErrorResponse struct {
	Status  string                 `json:"status"`
	Code    ServiceErrorStatusCode `json:"code"`
	ReqID   string                 `json:"req_id"`
	Message string                 `json:"message"`
}

func ResponseWithInternalServerError(
	ctx context.Context,
	w http.ResponseWriter,
	code ServiceErrorStatusCode,
	message string,
	items ...interface{},
) {
	ResponseWithError(ctx, w, http.StatusInternalServerError, code, message, items...)
}

func ResponseWithError(
	ctx context.Context,
	w http.ResponseWriter,
	httpCode int,
	code ServiceErrorStatusCode,
	message string,
	items ...interface{},
) {
	tracing.TagErrorMessage(ctx, message)
	reqID := ctxutil.GetRequestID(ctx)
	logger := ctxutil.GetLogger(ctx)
	msg := fmt.Sprintf(message, items...)
	response := ErrorResponse{
		Status:  code.String(),
		Code:    code,
		ReqID:   reqID,
		Message: msg,
	}
	body, err := json.Marshal(response)
	if err != nil {
		logger.Error(fmt.Sprintf("Unable to build response body for error (%v; %s): %s", code, msg, err.Error()),
			ctxutil.GetStoredFields(ctx)...)
	}

	http.Error(w, string(body), httpCode)
}

func LogErrorAndDie(ctx context.Context, w http.ResponseWriter, code ServiceErrorStatusCode, err error, safeMessage string) {
	logger := ctxutil.GetLogger(ctx)
	logger.Error(fmt.Sprintf("[%s] %s. ERR: %s", ctxutil.GetRequestID(ctx), safeMessage, err.Error()),
		ctxutil.GetStoredFields(ctx)...)
	ResponseWithInternalServerError(ctx, w, code, safeMessage)
}

func (c ServiceErrorStatusCode) String() string {
	switch c {
	case ServiceErrorStatusCodeIncorrectFormat:
		return "incorrect format"
	case ServiceErrorStatusCodeProxyCall:
		return "bad request"
	case ServiceErrorStatusCodeInfra:
		return "internal error"
	case ServiceErrorStatusCodeAuth:
		return "authorization failed"
	}
	return ""
}
