package api

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/unistat"
	v1 "a.yandex-team.ru/mail/payments-sdk-backend/internal/api/v1"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
	"net/http"
)

func NewRouter(logger log.Logger, bb blackbox.Client, checkAuth bool, server *Server) http.Handler {
	builder := routerBuilder{
		logger:    logger,
		bb:        bb,
		checkAuth: checkAuth,
		server:    server,
	}
	return builder.build()
}

type routerBuilder struct {
	logger    log.Logger
	bb        blackbox.Client
	server    *Server
	checkAuth bool
}

func (b *routerBuilder) build() chi.Router {
	r := chi.NewRouter()
	b.middlewares(r)
	b.unistat(r)
	b.ping(r)

	b.buildV1(r)

	return r
}

func (b *routerBuilder) buildV1(r chi.Router) {
	v1Router := chi.NewRouter()
	v1Router.Use(jaegerMiddleware())
	v1Router.Use(authMiddleware(b.bb, b.checkAuth))

	// Bad dependence
	apiV1 := v1.New(b.server.paymentService, b.server.bindingsService, b.server.nspkService)

	v1Router.Post("/payment_methods", apiV1.PaymentMethods)
	v1Router.Post("/init_payment", apiV1.InitPayment)
	v1Router.Post("/request_markup", apiV1.RequestMarkup)
	v1Router.Post("/verify_binding", apiV1.VerifyBinding)
	v1Router.Get("/nspk_bank_apps_ios", apiV1.BankAppsIOS)
	v1Router.Get("/nspk_bank_apps_android", apiV1.BankAppsAndroid)
	v1Router.Get("/nspk_bank_apps_common", apiV1.BankAppsCommon)
	v1Router.Get("/ping", func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		_, _ = w.Write([]byte(`{"status": "OK"}`))
	})
	r.Mount("/v1", v1Router)

	//Test ios Universal Links
	fs := http.FileServer(http.Dir("/etc/static/well-known"))
	r.Handle("/.well-known/*", http.StripPrefix("/.well-known/", fs))
}

func (b *routerBuilder) middlewares(r chi.Router) {
	r.Use(
		requestsStatMiddleware(b.server.serverMetrics),
		middleware.RequestID,
		middleware.RealIP,
		injectContext(b.logger),
		recoveryLogging(b.logger),
		jsonResponseFormat,
	)
}

func (b *routerBuilder) ping(r chi.Router) {
	r.Get("/ping", func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)
		_, _ = w.Write([]byte(`{"status": "OK"}`))
	})
}

func (b *routerBuilder) unistat(r chi.Router) {
	r.Get("/unistat", func(w http.ResponseWriter, r *http.Request) {
		ctx := r.Context()
		logger := ctxutil.GetLogger(ctx)

		result, err := unistat.MarshalJSON()
		if err != nil {
			logger.Error("Failed to build unistat response", log.Error(err))
			return
		}
		_, _ = w.Write(result)
	})

}
