package api

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/bindings"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/nspk"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/payment"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/stats"
	"context"
	"fmt"
	"net/http"
)

type ServerConfig struct {
	Host      string `config:"host"`
	Port      int    `config:"port"`
	CheckAuth bool   `config:"check_auth" yaml:"check_auth"`
}

type Server struct {
	config          *ServerConfig
	logger          log.Logger
	bb              blackbox.Client
	paymentService  payment.PaymentService
	bindingsService bindings.BindingsService
	nspkService     nspk.NspkService
	httpServer      *http.Server
	serverMetrics   *stats.Metrics
}

func NewServer(config *ServerConfig,
	logger log.Logger,
	bb blackbox.Client,
	ps payment.PaymentService,
	bs bindings.BindingsService,
	ns nspk.NspkService,
	metrics *stats.Metrics,
) *Server {
	s := &Server{
		config:          config,
		logger:          logger.WithName("api"),
		bb:              bb,
		paymentService:  ps,
		bindingsService: bs,
		nspkService:     ns,
		serverMetrics:   metrics,
	}
	s.httpServer = &http.Server{
		Addr:    fmt.Sprintf("%s:%d", config.Host, config.Port),
		Handler: NewRouter(s.logger, s.bb, config.CheckAuth, s),
	}

	return s
}

func (s *Server) Run(ctx context.Context) (err error) {
	logger := s.logger
	ctx, cancel := context.WithCancel(ctx)

	go func() {
		defer cancel()

		logger.Info("Starting server", log.String("addr", s.httpServer.Addr))
		err = s.httpServer.ListenAndServe()
		logger.Info("Shutting down server")
	}()

	if <-ctx.Done(); err != nil {
		logger.Debug("Server failed with error", log.Error(err))
		return
	}

	return
}
