package v1

import (
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/api/httputil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/models"
	paymentservice "a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/payment"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"regexp"
)

const SdkForceCvvHeader = "X-SDK-Force-CVV"

var emailValidationRegex = regexp.MustCompile("^[a-zA-Z0-9.!#$%&'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$")

type initPaymentRequest struct {
	Email      string `json:"email"`
	Token      string `json:"token"`
	TurboappID string `json:"turboapp_id"`
	Credit     bool   `json:"credit"`
}

func (r *initPaymentRequest) Validate() (err error) {
	if len(r.Token) == 0 {
		err = errors.New("empty token")
		return
	}

	if len(r.Email) > 0 && !emailValidationRegex.MatchString(r.Email) {
		err = errors.New("bad email")
		return
	}

	return
}

type initPaymentResponse struct {
	Status string `json:"status"`
	models.Payment
}

func (api *V1Api) InitPayment(w http.ResponseWriter, r *http.Request) {
	ctx := r.Context()
	ctx = ctxutil.WithForceCvv(ctx, r.Header.Get(SdkForceCvvHeader))
	logger := ctxutil.GetLogger(ctx)

	uid := ctxutil.GetUID(ctx)
	serviceToken := ctxutil.GetServiceToken(ctx)

	tracing.SetOperationName(ctx, "Payment initialization")

	req := initPaymentRequest{}

	err := json.NewDecoder(r.Body).Decode(&req)
	if err != nil {
		httputil.ResponseWithError(
			ctx, w,
			http.StatusBadRequest,
			httputil.ServiceErrorStatusCodeIncorrectFormat,
			fmt.Sprintf("body seems to be malformed: %v", err),
		)
		return
	}
	tracing.TagPurchaseToken(ctx, req.Token)

	err = req.Validate()
	if err != nil {
		httputil.ResponseWithError(
			ctx, w,
			http.StatusBadRequest,
			httputil.ServiceErrorStatusCodeIncorrectFormat,
			fmt.Sprintf("body seems to be malformed: %v", err),
		)
		return
	}

	payment, err := api.ps.InitPayment(ctx, uid, serviceToken, req.Email, req.Token, req.TurboappID, req.Credit)

	switch err {
	case nil:
	case paymentservice.ErrInvalidServiceToken:
		httputil.ResponseWithError(ctx, w, http.StatusForbidden, httputil.ServiceErrorStatusCodeInfra, "forbidden")
		return
	default:
		httputil.ResponseWithInternalServerError(ctx, w, httputil.ServiceErrorStatusCodeInfra, "internal_error")
		return
	}

	logger.Info(
		fmt.Sprintf("InitPayment methods info %v", payment.UserPaymentMethods),
		ctxutil.GetStoredFields(ctx)...,
	)
	resp := initPaymentResponse{
		Status:  "success",
		Payment: payment,
	}

	httputil.SuccessResponse(w, &resp, ctxutil.GetLogger(ctx))
}
