package v1

import (
	"a.yandex-team.ru/library/go/valid"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/api/httputil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/models"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"encoding/json"
	"fmt"
	"net/http"
	"reflect"
)

type RequestMarkupRequest struct {
	Token         string                  `json:"token" valid:"token"`
	SpasiboAmount string                  `json:"spasibo_amount" valid:"spasibo_amount"`
	Environment   models.OrderEnvironment `json:"environment"`
}

func (reqData *RequestMarkupRequest) Validate() error {
	vctx := valid.NewValidationCtx()
	vctx.Add("token", func(value reflect.Value, _ string) error {
		if value.String() == "" {
			return valid.ErrEmptyString
		}
		return nil
	})
	vctx.Add("spasibo_amount", valid.WrapValidator(valid.Double))
	return valid.Struct(vctx, reqData)
}

func (api *V1Api) RequestMarkup(w http.ResponseWriter, r *http.Request) {
	ctx := r.Context()
	uid := ctxutil.GetUID(ctx)
	serviceToken := ctxutil.GetServiceToken(ctx)
	logger := ctxutil.GetLogger(ctx)

	tracing.SetOperationName(ctx, "Request for Spasibo markup")

	req := RequestMarkupRequest{}
	err := json.NewDecoder(r.Body).Decode(&req)
	if err != nil {
		httputil.ResponseWithInternalServerError(ctx, w, httputil.ServiceErrorStatusCodeIncorrectFormat,
			fmt.Sprintf("body seems to be malformed: %v", err))
		return
	}
	tracing.TagPurchaseToken(ctx, req.Token)

	err = api.ps.RequestMarkup(ctx, uid, req.Token, req.SpasiboAmount, serviceToken, req.Environment)
	if err != nil {
		httputil.ResponseWithInternalServerError(ctx, w, httputil.ServiceErrorStatusCodeInfra, "internal error")
		return
	}

	httputil.SuccessResponse(w, &RequestMarkupResponse{Status: "success"}, logger)
}

type RequestMarkupResponse struct {
	Status    string `json:"status"`
	ErrorCode string `json:"error_code,omitempty"`
}
