package internal

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/api"
	cfg "a.yandex-team.ru/mail/payments-sdk-backend/internal/config"
	bindings "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/bindings/impl"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/blackbox"
	nspk_client "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/nspk/impl"
	trust "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/trust/impl"
	yapay "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/yapay/impl"
	bs "a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/bindings/provider"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/models"
	nspk_cache "a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/nspk/cache"
	nspk_provider "a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/nspk/provider"
	payment "a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/payment/provider"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/stats"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"context"
	"github.com/go-resty/resty/v2"
	"github.com/uber/jaeger-client-go"
	"go.uber.org/zap/zapcore"
	"io"
	"os"

	jaegerconfig "github.com/uber/jaeger-client-go/config"
)

type Application struct {
	config       *cfg.Config
	logger       log.Logger
	server       *api.Server
	jaegerCloser io.Closer
}

func GetTvmClient(config *cfg.Config, logger log.Logger) (*tvmtool.Client, error) {
	var opts []tvmtool.Option
	if config.TVM.SRC != "" {
		opts = append(opts, tvmtool.WithSrc(config.TVM.SRC))
	}

	if _, ok := os.LookupEnv("DEPLOY_TVM_TOOL_URL"); ok {
		logger.Info("Creating TVM deploy client")
		return tvmtool.NewDeployClient(opts...)
	} else {
		logger.Info("Creating TVM qloud client")
		return tvmtool.NewQloudClient(opts...)
	}
}

func newApp() (app *Application, err error) {
	config := cfg.Config{}
	logger, err := initLogger()

	if err != nil {
		return
	}

	if err = cfg.LoadConfig(&config, cfg.GetRunningEnvironment(), logger); err != nil {
		return
	}

	logger.Info("Initializing Jaeger")
	jaegerCloser, err := initJaeger(logger)
	if err != nil {
		logger.Errorf("Unable to initialize Jaeger: %v", err.Error())
		return
	}

	logger.Info("Initializing TVM client")
	tvmClient, err := GetTvmClient(&config, logger)
	if err != nil {
		logger.Error("Failed to initialize TVM client", log.Error(err))
		return
	}

	logger.Info("Initializing BlackBox client")
	bbMetrics := stats.NewMetrics("blackbox")
	bb, err := blackBoxClientFromEnvironment(cfg.GetRunningEnvironment(), bbMetrics, logger, tvmClient)
	if err != nil {
		logger.Error("Failed to initialize BlackBox client", log.Error(err))
		return
	}
	tracingBlackbox := blackbox.TracingClient{RealClient: bb}

	logger.Info("Initializing Trust Client")
	trustClient, err := trust.NewClient(
		&config.Trust,
		logger.WithName("trust-client"),
		tvmClient,
		stats.NewMetrics("trust"),
	)
	if err != nil {
		logger.Error("Failed to initialize Trust Client", log.Error(err))
		return
	}

	logger.Info("Initializing SandboxTrust Client")
	sandboxTrustClient, err := trust.NewClient(
		&config.SandboxTrust,
		logger.WithName("sandbox-trust-client"),
		tvmClient,
		stats.NewMetrics("sandbox-trust"),
	)
	if err != nil {
		logger.Error("Failed to initialize SandboxTrust Client", log.Error(err))
		return
	}

	logger.Info("Initializing YaPay client")
	yapayClient, err := yapay.NewClient(
		&config.YaPay,
		logger.WithName("yapay-client"),
		tvmClient,
		stats.NewMetrics("yapay"),
	)
	if err != nil {
		logger.Error("Failed to initialize YaPay Client", log.Error(err))
		return
	}

	logger.Info("Initializing Bindings client")
	bindingsClient, err := bindings.NewClient(
		&config.Bindings,
		logger.WithName("bindings-client"),
		tvmClient,
		stats.NewMetrics("bindings"),
	)
	if err != nil {
		logger.Error("Failed to initialize Bindings Client", log.Error(err))
		return
	}

	logger.Info("Initializing NSPK client")
	nspkClient, err := nspk_client.NewClient(
		&config.Nspk,
		logger.WithName("nspk-client"),
		tvmClient,
		stats.NewMetrics("nspk"),
	)
	if err != nil {
		logger.Error("Failed to initialize NSPK Client", log.Error(err))
		return
	}

	tracingTrustClient := trust.TracingClient{RealClient: trustClient}

	// Initialize business logic service
	paymentService, err := payment.NewService(&tracingTrustClient, sandboxTrustClient, yapayClient, logger.WithName("payment-service"),
		&config.Payment, models.GetResourceProvider())
	if err != nil {
		logger.Error("Failed to initialize Payment Service", log.Error(err))
		return
	}

	bindingsService, err := bs.NewService(bindingsClient, logger.WithName("bindings-service"))
	if err != nil {
		logger.Error("Failed to initialize Bindings Service", log.Error(err))
		return
	}

	nspkCache := nspk_cache.Cache{}
	err = nspk_cache.LoadCache(&nspkCache, logger.WithName("nspk-cache"))
	if err != nil {
		logger.Error("Failed to initialize NSPK cache", log.Error(err))
		return
	}

	nspkService, err := nspk_provider.NewService(nspkClient, logger.WithName("nspk-service"), &nspkCache)
	if err != nil {
		logger.Error("Failed to initialize NSPK Service", log.Error(err))
		return
	}

	serverMetrics := stats.NewMetrics("server")

	server := api.NewServer(&config.HTTP, logger, &tracingBlackbox, paymentService, bindingsService, nspkService, serverMetrics)
	app = &Application{config: &config, logger: logger, server: server, jaegerCloser: jaegerCloser}

	return
}

func blackBoxClientFromEnvironment(env cfg.RunningEnvironment, metrics *stats.Metrics, logger log.Logger, tvm tvm.Client) (*httpbb.Client, error) {
	logger = logger.WithName("bb")

	restyClient := resty.New()
	restyClient.OnAfterResponse(func(client *resty.Client, response *resty.Response) error {
		metrics.CountStatusCode(response.StatusCode())
		metrics.UpdateRequestTime(response.Time().Seconds())
		return nil
	})
	switch env {
	case cfg.RunningEnvironmentTest:
		return httpbb.NewClientWithResty(
			httpbb.TestEnvironment,
			restyClient,
			httpbb.WithLogger(logger.Structured()),
			httpbb.WithTVM(tvm),
		)
	case cfg.RunningEnvironmentProd:
		return httpbb.NewClientWithResty(
			httpbb.ProdEnvironment,
			restyClient,
			httpbb.WithLogger(logger.Structured()),
			httpbb.WithTVM(tvm),
		)
	case cfg.RunningEnvironmentMimino:
		return httpbb.NewClientWithResty(
			httpbb.MiminoEnvironment,
			restyClient,
			httpbb.WithLogger(logger.Structured()),
			httpbb.WithTVM(tvm),
		)
	}
	logger.Errorf("unknown environment: %v. Assuming Test", env)
	return httpbb.NewClientWithResty(httpbb.TestEnvironment, restyClient, httpbb.WithLogger(logger.Structured()))
}

func initLogger() (logger log.Logger, err error) {
	logLevelEnv, found := os.LookupEnv("LOG_LEVEL")
	logLevel := log.InfoLevel
	if found {
		logLevel, err = log.ParseLevel(logLevelEnv)
		if err != nil {
			logLevel = log.InfoLevel
		}
	}

	loggerType, _ := os.LookupEnv("LOG_LOGGER")
	if loggerType == "console" {
		logConfig := zap.ConsoleConfig(logLevel)
		logConfig.EncoderConfig.EncodeLevel = zapcore.CapitalColorLevelEncoder
		return zap.New(logConfig)
	}
	if loggerType == "yadeploy" {
		return zap.NewDeployLogger(logLevel)
	}
	return zap.NewQloudLogger(logLevel)
}

func initJaeger(logger log.Logger) (io.Closer, error) {
	jaegerConfig := jaegerconfig.Configuration{
		Sampler: &jaegerconfig.SamplerConfig{Type: jaeger.SamplerTypeConst, Param: 1},
		Reporter: &jaegerconfig.ReporterConfig{
			LogSpans:           true,
			LocalAgentHostPort: "localhost:6831",
		},
	}
	spanLogger := jaegerconfig.Logger(tracing.LoggerWrapper{Logger: logger})
	return jaegerConfig.InitGlobalTracer("payment-sdk-backend", spanLogger)
}

func Run() {
	app, err := newApp()
	if err != nil {
		panic(err)
	}

	ctx := context.Background()
	if err = app.server.Run(ctx); err != nil {
		panic(err)
	}

	if err = app.jaegerCloser.Close(); err != nil {
		panic(err)
	}
}
