package config

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/api"
	bindings "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/bindings/impl"
	nspk "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/nspk/impl"
	trust "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/trust/impl"
	yapay "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/yapay/impl"
	payservice "a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/payment/provider"
	"context"
	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"
	"os"
)

type Config struct {
	// Server settings
	HTTP api.ServerConfig `yaml:"http"`

	// TVM global settings
	TVM struct {
		SRC string `yaml:"src"`
	} `yaml:"tvm"`

	// Interaction clients settings
	Trust        trust.Config    `yaml:"trust"`
	SandboxTrust trust.Config    `yaml:"sandbox_trust"`
	YaPay        yapay.Config    `yaml:"yapay"`
	Bindings     bindings.Config `yaml:"bindings"`
	Nspk         nspk.Config     `yaml:"nspk"`

	// Payment business logic settings
	Payment payservice.Config `yaml:"payment"`
}

const defaultConfigPathTest = "/etc/payments-sdk-backend/testing.yaml"
const defaultConfigPathMimino = "/etc/payments-sdk-backend/testing-mimino.yaml"
const defaultConfigPathProd = "/etc/payments-sdk-backend/production.yaml"

func LoadConfig(cfg *Config, env RunningEnvironment, logger log.Logger) error {
	// TODO: config overloading

	configPath, ok := os.LookupEnv("CONFIG_PATH")
	if !ok {
		switch env {
		case RunningEnvironmentTest:
			configPath = defaultConfigPathTest
		case RunningEnvironmentMimino:
			configPath = defaultConfigPathMimino
		case RunningEnvironmentProd:
			configPath = defaultConfigPathProd
		}
	}

	logger.Info("Loading config file", log.String("path", configPath))
	loader := confita.NewLoader(file.NewBackend(configPath))
	err := loader.Load(context.Background(), cfg)
	if err != nil {
		logger.Error("Failed to load config file", log.String("path", configPath), log.Error(err))
		return err
	}

	return nil
}

type RunningEnvironment int

const (
	RunningEnvironmentTest   RunningEnvironment = iota
	RunningEnvironmentProd   RunningEnvironment = iota
	RunningEnvironmentMimino RunningEnvironment = iota
)

func GetRunningEnvironment() RunningEnvironment {
	if env, ok := os.LookupEnv("ENVIRONMENT_TYPE"); ok {
		switch env {
		case "testing", "payments-sdk-backend-testing":
			return RunningEnvironmentTest
		case "testing-mimino", "payments-sdk-backend-testing-mimino":
			return RunningEnvironmentMimino
		case "production", "payments-sdk-backend-production":
			return RunningEnvironmentProd
		default:
			return RunningEnvironmentTest
		}
	}
	return RunningEnvironmentTest
}
