package impl

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/bindings"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/stats"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"context"
	"encoding/json"
	"fmt"
	"github.com/go-resty/resty/v2"
	"time"
)

const (
	// AuthHeaderKeyXUid is a header key of UID
	AuthHeaderKeyXUid = "X-Uid"
	// AuthHeaderKeyXServiceToken is a header key of Service Token
	AuthHeaderKeyXServiceToken = "X-Service-Token"
)

type Config struct {
	BaseURL string `config:"base_url" yaml:"base_url"`
	TVMDest string `config:"tvm_dst" yaml:"tvm_dst"`
	Retries struct {
		Count       int `config:"count"`
		WaitTime    int `config:"wait_time" yaml:"wait_time"`
		MaxWaitTime int `config:"max_wait_time" yaml:"max_wait_time"`
	} `config:"retries"`
	Debug bool `config:"debug"`
}

type Client struct {
	config     *Config
	httpClient *resty.Client
	logger     log.Logger
	tvm        tvm.Client
	metrics    *stats.Metrics
}

var _ bindings.Client = &Client{}

func NewClient(config *Config, logger log.Logger, tvm tvm.Client, metrics *stats.Metrics) (*Client, error) {
	logger = logger.WithName("bindings-client")

	httpClient := resty.New()
	httpClient.SetLogger(logger.WithName("http"))

	httpClient.SetRetryCount(config.Retries.Count)
	if config.Retries.WaitTime > 0 {
		httpClient.SetRetryWaitTime(time.Duration(config.Retries.WaitTime) * time.Millisecond)
	}
	if config.Retries.MaxWaitTime > 0 {
		httpClient.SetRetryMaxWaitTime(time.Duration(config.Retries.MaxWaitTime) * time.Millisecond)
	}
	httpClient.AddRetryCondition(func(r *resty.Response, err error) bool {
		if err == nil {
			// If retries are enabled, we need to increment metrics here
			metrics.CountStatusCode(r.StatusCode())
			metrics.UpdateRequestTime(r.Time().Seconds())
		}
		//	if there is any connection problem or HTTP 5xx
		if err != nil || 500 <= r.StatusCode() {
			return true
		}

		return false
	})
	httpClient.EnableTrace()
	httpClient.SetDebug(config.Debug)

	return &Client{
		config:     config,
		httpClient: httpClient,
		logger:     logger,
		tvm:        tvm,
		metrics:    metrics,
	}, nil
}

type response interface {
	GetStatus() string
	GetMessage() string
}

func (c *Client) makeRequest(ctx context.Context, req interactions.Request, result response) (err error) {
	// TODO: Make base client
	url, err := interactions.MakeURL(c.config.BaseURL, req.APIMethod)
	if err != nil {
		// it mustn't happen in runtime
		panic(err)
	}

	span, ctx := tracing.StartSpanForRequest(ctx, url, req)
	defer span.Finish()

	serviceTicket, err := c.tvm.GetServiceTicketForAlias(ctx, c.config.TVMDest)

	if err != nil {
		msg := "Failed to get TVM"
		c.logger.Error(msg, append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	r := c.httpClient.R().
		SetHeader("X-Ya-Service-Ticket", serviceTicket).
		SetResult(result).
		SetError(result)

	if req.Headers != nil {
		r.SetHeaders(*req.Headers)
	}

	if req.Params != nil {
		r.SetQueryParams(*req.Params)
	}

	var resp *resty.Response
	switch req.Method {
	case interactions.HTTPPost:
		resp, err = r.SetBody(req.Body).Post(url)
	case interactions.HTTPGet:
		resp, err = r.Get(url)
	default:
		panic("unsupported method")
	}

	if err != nil {
		msg := "Bindings request failed"
		c.logger.Error(msg, append(ctxutil.GetStoredFields(ctx), log.Error(err), log.String("url", url))...)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	// If retries are disabled, increment metrics here
	if c.config.Retries.Count == 0 {
		c.metrics.UpdateRequestTime(resp.Time().Seconds())
		c.metrics.CountStatusCode(resp.StatusCode())
	}

	// Binding API returns wrong Content-Type header, so resty fails
	err = json.Unmarshal(resp.Body(), result)

	if resp.StatusCode() != 200 || err != nil {
		err = fmt.Errorf("bindings error, message: %s", result.GetMessage())

		msg := "Bindings request failed"
		c.logger.Info(msg,
			append(ctxutil.GetStoredFields(ctx), log.Error(err), log.String("url", url))...,
		)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	c.logger.Info("Bindings request finished",
		append(
			ctxutil.GetStoredFields(ctx),
			log.Int("status", resp.StatusCode()),
			log.Float64("response_time", resp.Time().Seconds()),
			log.String("url", url),
		)...,
	)

	return
}
