package impl

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/nspk"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/stats"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"context"
	"fmt"
	"github.com/go-resty/resty/v2"
	"time"
)

type Config struct {
	GozoraURL string `config:"gozora_url" yaml:"gozora_url"`
	BaseURL   string `config:"base_url" yaml:"base_url"`
	TVMDest   string `config:"tvm_dst" yaml:"tvm_dst"`
	Retries   struct {
		Count       int `config:"count" yaml:"count"`
		WaitTime    int `config:"wait_time" yaml:"wait_time"`
		MaxWaitTime int `config:"max_wait_time" yaml:"max_wait_time"`
	} `config:"retries"`
	Debug bool `config:"debug"`
}

type Client struct {
	config     *Config
	httpClient *resty.Client
	logger     log.Logger
	tvm        tvm.Client
	metrics    *stats.Metrics
}

var _ nspk.Client = &Client{}

func NewClient(config *Config, logger log.Logger, tvm tvm.Client, metrics *stats.Metrics) (*Client, error) {
	logger = logger.WithName("nspk-client")

	httpClient := resty.New()
	httpClient.SetLogger(logger.WithName("http"))

	httpClient.SetRetryCount(config.Retries.Count)
	if config.Retries.WaitTime > 0 {
		httpClient.SetRetryWaitTime(time.Duration(config.Retries.WaitTime) * time.Millisecond)
	}
	if config.Retries.MaxWaitTime > 0 {
		httpClient.SetRetryMaxWaitTime(time.Duration(config.Retries.MaxWaitTime) * time.Millisecond)
	}
	httpClient.AddRetryCondition(func(r *resty.Response, err error) bool {
		if err == nil {
			// If retries are enabled, we need to increment metrics here
			metrics.CountStatusCode(r.StatusCode())
			metrics.UpdateRequestTime(r.Time().Seconds())
		}
		//	if there is any connection problem or HTTP 5xx
		if err != nil || 500 <= r.StatusCode() {
			return true
		}

		return false
	})
	httpClient.EnableTrace()
	httpClient.SetDebug(config.Debug)

	return &Client{
		config:     config,
		httpClient: httpClient,
		logger:     logger,
		tvm:        tvm,
		metrics:    metrics,
	}, nil
}

type response interface{}

func (c *Client) makeRequest(ctx context.Context, req interactions.Request, result response) (err error) {
	url, err := interactions.MakeURL(c.config.BaseURL, req.APIMethod)
	if err != nil {
		// it mustn't happen in runtime
		panic(err)
	}

	span, ctx := tracing.StartSpanForRequest(ctx, url, req)
	defer span.Finish()

	serviceTicket, err := c.tvm.GetServiceTicketForAlias(ctx, c.config.TVMDest)

	if err != nil {
		msg := "Failed to get TVM"
		c.logger.Error(msg, append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	r := c.httpClient.R().
		SetHeader("X-Ya-Service-Ticket", serviceTicket).
		SetHeader("x-ya-dest-url", url).
		SetHeader("x-ya-client-id", "payback").
		SetResult(result).
		SetError(result)

	gozoraURL := c.config.GozoraURL

	var resp *resty.Response
	switch req.Method {
	case interactions.HTTPPost:
		resp, err = r.SetBody(req.Body).Post(gozoraURL)
	case interactions.HTTPGet:
		resp, err = r.Get(gozoraURL)
	default:
		panic("unsupported method")
	}

	if err != nil {
		msg := "NSPK request failed"
		c.logger.Warn(msg,
			append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	c.metrics.UpdateRequestTime(resp.Time().Seconds())
	c.metrics.CountStatusCode(resp.StatusCode())

	if resp.StatusCode() != 200 {
		err = fmt.Errorf("NSPK error. Code: %d", resp.StatusCode())

		msg := "NSPK request failed"
		c.logger.Warn(msg,
			append(ctxutil.GetStoredFields(ctx), log.Error(err))...,
		)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	c.logger.Info("NSPK request finished",
		append(
			ctxutil.GetStoredFields(ctx),
			log.Int("status", resp.StatusCode()),
			log.Float64("response_time", resp.Time().Seconds()),
		)...,
	)

	return
}
