package impl

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/trust"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/stats"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"context"
	"fmt"
	"github.com/go-resty/resty/v2"
	"time"
)

const (
	// AuthHeaderKeyXUid is a header key of UID
	AuthHeaderKeyXUid = "X-Uid"
	// AuthHeaderKeyXServiceToken is a header key of Service Token
	AuthHeaderKeyXServiceToken = "X-Service-Token"
)

type Config struct {
	BaseURL       string            `config:"base_url" yaml:"base_url"`
	PerServiceURL map[string]string `config:"per_service_url" yaml:"per_service_url"`
	TVMDest       string            `config:"tvm_dst" yaml:"tvm_dst"`
	Retries       struct {
		Count       int `config:"count"`
		WaitTime    int `config:"wait_time" yaml:"wait_time"`
		MaxWaitTime int `config:"max_wait_time" yaml:"max_wait_time"`
	} `config:"retries"`
	Debug bool `config:"debug"`
}

type Client struct {
	config     *Config
	httpClient *resty.Client
	logger     log.Logger
	tvm        tvm.Client
	metrics    *stats.Metrics
}

var _ trust.Client = &Client{}

func NewClient(config *Config, logger log.Logger, tvm tvm.Client, metrics *stats.Metrics) (*Client, error) {
	logger = logger.WithName("trust-client")

	httpClient := resty.New()
	httpClient.SetLogger(logger.WithName("http"))

	httpClient.SetRedirectPolicy(tracing.TracingDelegateRedirectPolicy(resty.FlexibleRedirectPolicy(3)))
	httpClient.SetRetryCount(config.Retries.Count)
	if config.Retries.WaitTime > 0 {
		httpClient.SetRetryWaitTime(time.Duration(config.Retries.WaitTime) * time.Millisecond)
	}
	if config.Retries.MaxWaitTime > 0 {
		httpClient.SetRetryMaxWaitTime(time.Duration(config.Retries.MaxWaitTime) * time.Millisecond)
	}
	httpClient.AddRetryCondition(func(r *resty.Response, err error) bool {
		if err == nil {
			// If retries are enabled, we need to increment metrics here
			metrics.CountStatusCode(r.StatusCode())
			metrics.UpdateRequestTime(r.Time().Seconds())
		}
		//	if there is any connection problem or HTTP 5xx
		if err != nil || 500 <= r.StatusCode() {
			return true
		}

		return false
	})
	httpClient.EnableTrace()
	httpClient.SetDebug(config.Debug)

	return &Client{
		config:     config,
		httpClient: httpClient,
		logger:     logger,
		tvm:        tvm,
		metrics:    metrics,
	}, nil
}

type response interface {
	GetStatus() string
	GetMessage() string
}

func (c *Client) makeRequest(ctx context.Context, req interactions.Request, result response) (err error) {
	// TODO: Make base client
	baseURL := c.config.BaseURL
	serviceToken := ctxutil.GetServiceToken(ctx)
	if serviceURL, found := c.config.PerServiceURL[serviceToken]; found {
		baseURL = serviceURL
	}
	url, err := interactions.MakeURL(baseURL, req.APIMethod)
	if err != nil {
		// it mustn't happen in runtime
		panic(err)
	}

	span, ctx := tracing.StartSpanForRequest(ctx, url, req)
	defer span.Finish()

	r := c.httpClient.R()

	if c.config.TVMDest != "" {
		var serviceTicket string
		serviceTicket, err = c.tvm.GetServiceTicketForAlias(ctx, c.config.TVMDest)
		if err != nil {
			msg := "Failed to get TVM"
			c.logger.Error(msg, append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
			tracing.TagErrorWithMessage(ctx, msg, err)
			return
		}

		r = r.SetHeader("X-Ya-Service-Ticket", serviceTicket)
	}

	r = r.SetResult(result).SetError(result)

	if req.Headers != nil {
		r.SetHeaders(*req.Headers)
	}

	if req.Params != nil {
		r.SetQueryParams(*req.Params)
	}

	var resp *resty.Response
	switch req.Method {
	case interactions.HTTPPost:
		resp, err = r.SetBody(req.Body).Post(url)
	case interactions.HTTPGet:
		resp, err = r.Get(url)
	default:
		panic("unsupported method")
	}

	if err != nil {
		msg := "Trust request failed"
		c.logger.Error(
			msg,
			append(ctxutil.GetStoredFields(ctx), log.Error(err), log.String("url", url))...,
		)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	// If retries are disabled, increment metrics here
	if c.config.Retries.Count == 0 {
		c.metrics.UpdateRequestTime(resp.Time().Seconds())
		c.metrics.CountStatusCode(resp.StatusCode())
	}

	if result.GetStatus() != "success" {
		err = fmt.Errorf("trust error. Message: %s", result.GetMessage())

		msg := "Trust request failed"
		c.logger.Info(msg,
			append(ctxutil.GetStoredFields(ctx), log.Error(err), log.String("url", url))...,
		)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	c.logger.Info("Trust request finished",
		append(
			ctxutil.GetStoredFields(ctx),
			log.Int("status", resp.StatusCode()),
			log.Float64("response_time", resp.Time().Seconds()),
			log.String("url", url),
		)...,
	)

	return
}
