package trust

import (
	"context"
	"fmt"
	"strings"
)

type Client interface {
	GetPaymentStatus(ctx context.Context, purchaseToken string, serviceToken string, uid uint64) (PaymentStatus, error)
	GetPaymentMethods(ctx context.Context, serviceToken string, uid uint64) (PaymentMethods, error)
	GetAFPaymentMethods(ctx context.Context, purchaseToken string, serviceToken string, uid uint64) (AFPaymentMethods, error)
	StartPayment(ctx context.Context, purchaseToken string, serviceToken string, uid uint64, credit bool) (StartPaymentInfo, error)
	RequestMarkup(ctx context.Context, purchaseToken string, serviceToken string, uid uint64, spasiboAmount string) error
}

type StartPaymentInfo struct {
	Status        string `json:"status"`
	PaymentURL    string `json:"payment_url"`
	CreditFormURL string `json:"credit_form_url"`
}

type PaymentStatus struct {
	Status        string                       `json:"status"`
	PurchaseToken string                       `json:"purchase_token"`
	Total         string                       `json:"amount"`
	Currency      string                       `json:"currency"`
	Orders        []OrderItem                  `json:"orders"`
	PaymentMarkup map[string]map[string]string `json:"payment_markup"`
}

func (r PaymentStatus) String() string {
	return fmt.Sprintf("PaymentsStatus[Status=\"%s\"; Purchase Token=\"%s\"]", r.Status, r.PurchaseToken)
}

type OrderItem struct {
	OrderID         string `json:"order_id"`
	OrderTS         string `json:"order_ts"`
	ProductID       string `json:"product_id"`
	ProductType     string `json:"product_type"`
	ProductName     string `json:"product_name"`
	OriginalAmount  string `json:"orig_amount"`
	PaidAmount      string `json:"paid_amount"`
	CurrentQuantity string `json:"current_qty"`
}

type PaymentMethods struct {
	Status                string                  `json:"status"`
	EnabledPaymentMethods []EnabledPaymentMethods `json:"enabled_payment_methods"`
	BoundPaymentMethods   []BoundPaymentMethod    `json:"bound_payment_methods"`
}

type BoundPaymentMethod struct {
	RegionID                    int64        `json:"region_id"`
	PaymentMethod               string       `json:"payment_method"`
	BindingTS                   string       `json:"binding_ts"`
	RecommendedVerificationType string       `json:"recommended_verification_type"`
	Aliases                     []string     `json:"aliases"`
	Expired                     bool         `json:"expired"`
	CardBank                    string       `json:"card_bank"`
	System                      string       `json:"system"`
	ID                          string       `json:"id"`
	CardCountry                 string       `json:"card_country"`
	PaymentSystem               string       `json:"payment_system"`
	CardLevel                   string       `json:"card_level"`
	Holder                      string       `json:"holder"`
	BindingSystems              []string     `json:"binding_systems"`
	Account                     string       `json:"account"`
	CardID                      string       `json:"card_id"`
	VerifyCvv                   bool         `json:"verify_cvv"`
	IsSpasibo                   int64        `json:"is_spasibo"`
	LastPaid                    int64        `json:"last_paid"`
	LastServicePaid             int64        `json:"last_service_paid"`
	PayerInfo                   *PayerInfo   `json:"payer_info"`
	PartnerInfo                 *PartnerInfo `json:"partner_info"`
}

type EnabledPaymentMethods struct {
	PaymentMethod  string   `json:"payment_method"`
	PaymentSystems []string `json:"payment_systems,omitempty"`
	Currency       string   `json:"currency"`
	FirmID         int64    `json:"firm_id"`
}

type AFBoundPaymentMethod struct {
	CardID   string `json:"card_id"`
	NeedCvv  bool   `json:"need_cvv"`
	Raise3ds bool   `json:"raise_3_ds"`
}

type AFPaymentMethods struct {
	Status         string                 `json:"status"`
	PaymentMethods []AFBoundPaymentMethod `json:"bound_payment_methods"`
}

func (pm PaymentMethods) String() string {
	paymentMethods := strings.Builder{}
	for i := 0; i < len(pm.BoundPaymentMethods); i++ {
		_, _ = paymentMethods.WriteString(pm.BoundPaymentMethods[i].ID)
		if i < len(pm.BoundPaymentMethods)-1 {
			paymentMethods.WriteString(", ")
		}
	}
	enabledMethods := strings.Builder{}
	for i := 0; i < len(pm.EnabledPaymentMethods); i++ {
		_, _ = enabledMethods.WriteString(pm.EnabledPaymentMethods[i].PaymentMethod)
		if i < len(pm.EnabledPaymentMethods)-1 {
			enabledMethods.WriteString(", ")
		}
	}
	return fmt.Sprintf(
		"Bound methods: [%v] Enabled methods: [%v]",
		paymentMethods.String(),
		enabledMethods.String(),
	)
}

// PayerInfo https://wiki.yandex-team.ru/users/amosov-f/familypay/#variantformataotvetaruchkilpmvtraste
type PayerInfo struct {
	UID        string     `json:"uid"`
	FamilyInfo FamilyInfo `json:"family_info"`
}

type FamilyInfo struct {
	FamilyID string `json:"family_id"`
	Expenses int    `json:"expenses"`
	Limit    int    `json:"limit"`
	Frame    string `json:"frame"`
	Currency string `json:"currency"`
}

// YaBank
type PartnerInfo struct {
	IsYabankCard      bool `json:"is_yabank_card"`
	IsYabankCardOwner bool `json:"is_yabank_card_owner"`
	IsFakeYabankCard  bool `json:"is_fake_yabank_card"`
}
