package impl

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/yapay"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/stats"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/tracing"
	"context"
	"fmt"
	"github.com/go-resty/resty/v2"
	"time"
)

type Config struct {
	BaseURL string `config:"base_url" yaml:"base_url"`
	TVMDest string `config:"tvm_dst" yaml:"tvm_dst"`
	Retries struct {
		Count       int `config:"count"`
		WaitTime    int `config:"wait_time" yaml:"wait_time"`
		MaxWaitTime int `config:"max_wait_time" yaml:"max_wait_time"`
	} `config:"retries"`
	Debug bool `config:"debug"`
}

type Client struct {
	config     *Config
	httpClient *resty.Client
	logger     log.Logger
	tvm        tvm.Client
	metrics    *stats.Metrics
}

var _ yapay.Client = &Client{}

func NewClient(config *Config, logger log.Logger, tvm tvm.Client, metrics *stats.Metrics) (*Client, error) {
	logger = logger.WithName("yapay-client")

	httpClient := resty.New()
	httpClient.SetLogger(logger.WithName("http"))

	httpClient.SetRetryCount(config.Retries.Count)
	if config.Retries.WaitTime > 0 {
		httpClient.SetRetryWaitTime(time.Duration(config.Retries.WaitTime) * time.Millisecond)
	}
	if config.Retries.MaxWaitTime > 0 {
		httpClient.SetRetryMaxWaitTime(time.Duration(config.Retries.MaxWaitTime) * time.Millisecond)
	}
	httpClient.AddRetryCondition(retryOn)
	httpClient.EnableTrace()
	httpClient.SetDebug(config.Debug)

	return &Client{
		config:     config,
		httpClient: httpClient,
		logger:     logger,
		tvm:        tvm,
		metrics:    metrics,
	}, nil
}

type yapayResponse interface {
	GetStatus() string
	GetMessage() string
	GetCode() int
}

func (c *Client) makeRequest(ctx context.Context, req interactions.Request, result yapayResponse) (err error) {
	url, err := interactions.MakeURL(c.config.BaseURL, req.APIMethod)
	if err != nil {
		// it mustn't happen in runtime
		panic(err)
	}

	span, ctx := tracing.StartSpanForRequest(ctx, url, req)
	defer span.Finish()

	serviceTicket, err := c.tvm.GetServiceTicketForAlias(ctx, c.config.TVMDest)

	if err != nil {
		msg := "Failed to get TVM"
		c.logger.Error(msg, append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	r := c.httpClient.R().
		SetHeader("X-Ya-Service-Ticket", serviceTicket).
		SetHeaders(*req.Headers).
		SetResult(result).
		SetError(result)

	var resp *resty.Response
	switch req.Method {
	case interactions.HTTPPost:
		resp, err = r.SetBody(req.Body).Post(url)
	case interactions.HTTPGet:
		resp, err = r.Get(url)
	default:
		panic("unsupported method")
	}

	if err != nil {
		msg := "YaPayment request failed"
		c.logger.Error(msg,
			append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	c.metrics.UpdateRequestTime(resp.Time().Seconds())
	c.metrics.CountStatusCode(resp.StatusCode())

	if result.GetStatus() != "success" {
		err = fmt.Errorf("YaPayment error. Code: %d. Message: %s",
			result.GetCode(), result.GetMessage())

		msg := "YaPayment request failed"
		c.logger.Info(msg,
			append(ctxutil.GetStoredFields(ctx), log.Error(err))...,
		)
		tracing.TagErrorWithMessage(ctx, msg, err)
		return
	}

	c.logger.Info("YaPayment request finished",
		append(
			ctxutil.GetStoredFields(ctx),
			log.Int("status", resp.StatusCode()),
			log.Float64("response_time", resp.Time().Seconds()),
		)...,
	)

	return
}

func retryOn(r *resty.Response, err error) bool {
	// TODO: add retry condition
	return false
}
