package models

import (
	"fmt"
	"strconv"
	"strings"
)

type PaymentSystem string

func (s PaymentSystem) String() string {
	return string(s)
}

type PaymentMethodType string

func (t PaymentMethodType) String() string {
	return string(t)
}

const (
	ApplePay  PaymentSystem = "ApplePay"
	GooglePay PaymentSystem = "GooglePay"

	SBPQR   PaymentMethodType = "sbp_qr"
	YaMoney PaymentMethodType = "yamoney_wallet"
)

type EnabledPaymentMethods struct {
	PaymentMethod  string   `json:"payment_method"`
	PaymentSystems []string `json:"payment_systems,omitempty"`
	Currency       string   `json:"currency"`
	FirmID         int64    `json:"firm_id"`
}

type BoundPaymentMethod struct {
	RegionID                    int64        `json:"region_id"`
	PaymentMethod               string       `json:"payment_method"`
	BindingTS                   string       `json:"binding_ts"`
	RecommendedVerificationType string       `json:"recommended_verification_type"`
	Aliases                     []string     `json:"aliases"`
	Expired                     bool         `json:"expired"`
	CardBank                    string       `json:"card_bank"`
	System                      string       `json:"system"`
	ID                          string       `json:"id"`
	CardCountry                 string       `json:"card_country"`
	PaymentSystem               string       `json:"payment_system"`
	CardLevel                   string       `json:"card_level"`
	Holder                      string       `json:"holder"`
	BindingSystems              []string     `json:"binding_systems"`
	Account                     string       `json:"account"`
	CardID                      string       `json:"card_id"`
	VerifyCvv                   bool         `json:"verify_cvv"`
	IsSpasibo                   int64        `json:"is_spasibo"`
	LastPaid                    int64        `json:"last_paid"`
	LastServicePaid             int64        `json:"last_service_paid"`
	PayerInfo                   *PayerInfo   `json:"payer_info"`
	PartnerInfo                 *PartnerInfo `json:"partner_info"`
}

type PayerInfo struct {
	UID        string     `json:"uid"`
	FamilyInfo FamilyInfo `json:"family_info"`
}

type PartnerInfo struct {
	IsYabankCard      bool `json:"is_yabank_card"`
	IsYabankCardOwner bool `json:"is_yabank_card_owner"`
	IsFakeYabankCard  bool `json:"is_fake_yabank_card"`
}

type FamilyInfo struct {
	FamilyID  string `json:"family_id"`
	Expenses  int    `json:"expenses"`
	Limit     int    `json:"limit"`
	Frame     string `json:"frame"`
	Currency  string `json:"currency"`
	Unlimited bool   `json:"unlimited"`
}

func (m *BoundPaymentMethod) MapBankName(mapping map[string]string) {
	bankName, found := mapping[strings.ToUpper(m.CardBank)]
	if found {
		m.CardBank = bankName
	} else {
		m.CardBank = "UnknownBank"
	}
}

type BoundPaymentMethods []BoundPaymentMethod

func (m BoundPaymentMethods) Len() int { return len(m) }

func (m BoundPaymentMethods) Less(i, j int) bool {
	a, _ := strconv.ParseFloat(m[i].BindingTS, 64)
	b, _ := strconv.ParseFloat(m[j].BindingTS, 64)
	return a < b
}

func (m BoundPaymentMethods) Swap(i, j int) {
	m[i], m[j] = m[j], m[i]
}

type UserPaymentMethods struct {
	GooglePaySupported    bool                    `json:"google_pay_supported"`
	ApplePaySupported     bool                    `json:"apple_pay_supported"`
	PaymentMethods        BoundPaymentMethods     `json:"payment_methods"`
	EnabledPaymentMethods []EnabledPaymentMethods `json:"enabled_payment_methods"`
}

func (pm *UserPaymentMethods) PaymentSystemEnabled(system PaymentSystem) bool {
	systemString := system.String()
	for _, e := range pm.EnabledPaymentMethods {
		for _, s := range e.PaymentSystems {
			if s == systemString {
				return true
			}
		}
	}
	return false
}

func (pm UserPaymentMethods) String() string {
	paymentMethods := strings.Builder{}
	for i := 0; i < len(pm.PaymentMethods); i++ {
		_, _ = paymentMethods.WriteString(pm.PaymentMethods[i].ID)
		if i < len(pm.PaymentMethods)-1 {
			paymentMethods.WriteString(", ")
		}
	}
	enabledMethods := strings.Builder{}
	for i := 0; i < len(pm.EnabledPaymentMethods); i++ {
		_, _ = enabledMethods.WriteString(pm.EnabledPaymentMethods[i].PaymentMethod)
		if i < len(pm.EnabledPaymentMethods)-1 {
			enabledMethods.WriteString(", ")
		}
	}
	return fmt.Sprintf(
		"Payment methods: [%v] Enabled methods: [%v] GooglePaySupported: %v ApplePaySupported: %v",
		paymentMethods.String(),
		enabledMethods.String(),
		pm.GooglePaySupported,
		pm.ApplePaySupported,
	)
}

type PaymentSystemsOptions struct {
	ApplePayEnabled  bool
	GooglePayEnabled bool
}
