package provider

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	interactions "a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/nspk"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/models"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/nspk"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/nspk/cache"
)

type NspkService struct {
	client interactions.Client
	logger log.Logger
	cache  *cache.Cache
}

var _ nspk.NspkService = &NspkService{}

func NewService(client interactions.Client, logger log.Logger, cache *cache.Cache) (*NspkService, error) {
	return &NspkService{client: client, logger: logger, cache: cache}, nil
}

func (n NspkService) GetBankAppsIOS(ctx context.Context) []models.NspkBankAppIOS {
	res, err := n.client.GetProxyAppC2BMembers(ctx)
	if err != nil {
		n.logger.Warn("Failed to load NSPK bank apps", log.String("type", "iOS"), log.Error(err))
		res = n.cache.DefaultResponseC2BMembers
	}
	apps := make([]models.NspkBankAppIOS, len(res.Dictionary))
	for i, app := range res.Dictionary {
		apps[i] = models.NspkBankAppIOS{
			BankName: app.BankName,
			LogoURL:  app.LogoURL,
			Schema:   app.Schema,
		}
	}
	return apps
}

func (n NspkService) GetBankAppsAndroid(ctx context.Context) []models.NspkBankAppAndroid {
	res, err := n.client.GetWellKnownAssetLinks(ctx)
	if err != nil {
		n.logger.Warn("Failed to load NSPK bank apps", log.String("type", "Android"), log.Error(err))
		res = n.cache.DefaultResponseDigitalAssetLinks
	}
	apps := make([]models.NspkBankAppAndroid, len(res))
	for i, app := range res {
		apps[i] = models.NspkBankAppAndroid{
			PackageName: app.Target.PackageName,
		}
	}
	return apps
}

func (n NspkService) GetBankAppsCommon(ctx context.Context) []models.NspkBankAppCommon {
	res, err := n.client.GetProxyAppC2BMembers(ctx)
	if err != nil {
		n.logger.Warn("Failed to load NSPK bank apps", log.String("type", "Common"), log.Error(err))
		res = n.cache.DefaultResponseC2BMembers
	}
	apps := make([]models.NspkBankAppCommon, len(res.Dictionary))
	for i, app := range res.Dictionary {
		apps[i] = models.NspkBankAppCommon{
			BankName:    app.BankName,
			LogoURL:     app.LogoURL,
			Schema:      app.Schema,
			PackageName: app.PackageName,
		}
	}
	return apps
}
