package provider

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/trust"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/models"
	paymentservice "a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/payment"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"context"
	"github.com/shopspring/decimal"
	"strings"
)

func makePayMethodMarkup(payMarkup map[string]map[string]string, logger log.Logger) map[string]string {
	payment := make(map[string]string)

	for _, methods := range payMarkup {
		for method, amount := range methods {
			amountDec, err := decimal.NewFromString(strings.TrimSpace(amount))
			if err != nil {
				logger.Warn("Total is empty for payment", log.String("method", method), log.String("amount", amount))
				continue
			}

			if val, ok := payment[method]; ok {
				valDec, _ := decimal.NewFromString(val)
				payment[method] = decimal.Sum(valDec, amountDec).String()
			} else {
				payment[method] = amountDec.String()
			}
		}
	}

	return payment
}

func shouldVerifyCvv(method models.BoundPaymentMethod, cfg *Config, xServiceToken string, sdkVersion string, cardMapping map[string]bool, forceCvv bool) (bool, error) {
	if forceCvv {
		return true, nil
	}

	service, err := cfg.GetServiceByToken(xServiceToken)
	if err != nil {
		return true, err
	}

	if sdkVersion == "" {
		return true, nil
	}

	checkCard, ok := cardMapping[method.CardID]
	if ok && !checkCard {
		return false, nil
	}

	if service != nil {
		return service.VerifyCvv, nil
	}
	return true, nil
}

func fixTotal(ctx context.Context, total string, logger log.Logger) string {
	if len(total) == 0 {
		logger.Warn("Total is empty for payment", ctxutil.GetStoredFields(ctx)...)
		total = "0.00"
	}
	return total
}

func fixCurrency(ctx context.Context, currency string, logger log.Logger) string {
	if len(currency) == 0 {
		logger.Warn("Currency is empty for payment", ctxutil.GetStoredFields(ctx)...)
		currency = "RUB"
	}
	return currency
}

func (s *PaymentService) InitPayment(ctx context.Context, uid uint64, serviceToken string,
	email string, token string, turboappID string, credit bool) (payment models.Payment, err error) {
	ctx = ctxutil.WithPayToken(ctx, token)
	payToken := models.PayToken(token)
	var purchaseToken string
	var paymentSystemsOptions = models.PaymentSystemsOptions{
		ApplePayEnabled:  true,
		GooglePayEnabled: true,
	}

	switch payToken.Type() {
	case models.TokenTypeYaPayment:
		orderData, err := s.yapayClient.StartOrder(ctx, uid, token, turboappID, email)
		if err != nil {
			return payment, err
		}
		purchaseToken = orderData.PurchaseToken

		if s.config.PartnerPaymentOptionsEnabled { // Чтобы можно было катиться независимо от payments-backend
			paymentSystemsOptions = models.PaymentSystemsOptions(orderData.PaymentSystemsOptions)
		}

		payment.LicenseURL = s.config.LicenceURL
		payment.Environment = models.OrderEnvironment(orderData.Environment)
		payment.AcquirerType = orderData.AcquirerType
		serviceToken = s.config.GetServiceToken(payment.Environment, orderData.AcquirerType)
		payment.Merchant = models.Merchant{
			Name:         orderData.Merchant.Name,
			OGRN:         orderData.Merchant.OGRN,
			ScheduleText: orderData.Merchant.ScheduleText,
			LegalAddress: models.LegalAddress{
				City:    orderData.Merchant.LegalAddress.City,
				Country: orderData.Merchant.LegalAddress.Country,
				Home:    orderData.Merchant.LegalAddress.Home,
				Street:  orderData.Merchant.LegalAddress.Street,
				Zip:     orderData.Merchant.LegalAddress.Zip,
			},
		}
	case models.TokenTypeTrust:
		purchaseToken = token
		payment.Environment = models.ProductionOrder
	}

	// Since some orders are now created in a separate trust environment, we must use appropriate client
	trustClient := s.getTrustClient(payment.Environment)

	if len(serviceToken) == 0 {
		err = paymentservice.ErrInvalidServiceToken
		return
	}

	ctx = ctxutil.WithServiceToken(ctx, serviceToken)

	service, err := s.config.GetServiceByToken(serviceToken)
	useAntiFraud := err == nil && service != nil && service.CheckAntiFraud

	var afm chan trust.AFPaymentMethods
	// Parallel request to AF
	if useAntiFraud {
		afm = make(chan trust.AFPaymentMethods, 1)
		go func() {
			s.logger.Info("Fetch AF payment methods data", ctxutil.GetStoredFields(ctx)...)
			r, err := s.trustClient.GetAFPaymentMethods(ctx, purchaseToken, serviceToken, uid)
			if err != nil {
				// We ignore AF errors and use default CVV behaviour (raise CVV)
				s.logger.Error("Failed to fetch AF payment methods data",
					append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
			}
			afm <- r
		}()
	}

	// Parallel request to payment methods
	type pmResult struct {
		res models.UserPaymentMethods
		err error
	}
	methods := make(chan pmResult, 1)
	go func() {
		s.logger.Info("Fetch payment methods", ctxutil.GetStoredFields(ctx)...)
		r, err := s.PaymentMethods(ctx, uid, serviceToken, paymentSystemsOptions, payment.Environment)

		if err != nil {
			s.logger.Error(
				"Failed to fetch payment methods",
				append(ctxutil.GetStoredFields(ctx), log.Error(err))...,
			)
		}

		methods <- pmResult{r, err}
	}()

	paymentStatus, err := trustClient.GetPaymentStatus(ctx, purchaseToken, serviceToken, uid)
	if err != nil {
		s.logger.Error("Failed to get payment status", append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
		return
	}

	startPaymentInfo, err := trustClient.StartPayment(ctx, purchaseToken, serviceToken, uid, credit)
	if err != nil {
		s.logger.Error("Failed to start payment", append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
		return
	}

	methodsRes := <-methods
	if methodsRes.err != nil {
		err = methodsRes.err
		return payment, err
	}

	payment.Token = purchaseToken
	payment.PaymentMethods = methodsRes.res.PaymentMethods
	payment.EnabledPaymentMethods = methodsRes.res.EnabledPaymentMethods
	payment.ApplePaySupported = methodsRes.res.ApplePaySupported
	payment.GooglePaySupported = methodsRes.res.GooglePaySupported
	payment.PayMethodMarkup = makePayMethodMarkup(paymentStatus.PaymentMarkup, s.logger)
	payment.PaymentURL = startPaymentInfo.PaymentURL
	payment.CreditFormURL = startPaymentInfo.CreditFormURL

	afMethodsCvvMapping := make(map[string]bool)
	// Applying AF rules on payment methods
	if useAntiFraud {
		for _, paymentMethod := range (<-afm).PaymentMethods {
			afMethodsCvvMapping[paymentMethod.CardID] = paymentMethod.NeedCvv
		}
	}

	for idx, m := range payment.PaymentMethods {
		verify, err := shouldVerifyCvv(m, s.config, serviceToken, ctxutil.GetSdkVersion(ctx), afMethodsCvvMapping, ctxutil.GetForceCvv(ctx))
		if err != nil {
			s.logger.Error("Failed to map verify methods", append(ctxutil.GetStoredFields(ctx), log.Error(err))...)
			return payment, err
		}
		payment.PaymentMethods[idx].VerifyCvv = verify
	}

	// Google pay data
	var gpayData *models.GPayData = nil

	if payment.GooglePaySupported {
		switch payToken.Type() {
		case models.TokenTypeYaPayment:
			gpayData = &models.GPayData{
				Gateway:           models.GPayGatewayYaPayments,
				GatewayMerchantID: s.config.GooglePay.Gateway.YaPay,
			}
		case models.TokenTypeTrust:
			gpayData = &models.GPayData{
				Gateway:           models.GPayGatewayTrust,
				GatewayMerchantID: serviceToken,
			}
		}
		payment.GooglePay = gpayData
	}

	payment.Total = fixTotal(ctx, paymentStatus.Total, s.logger)
	payment.Currency = fixCurrency(ctx, paymentStatus.Currency, s.logger)

	s.logger.Debug("Done running InitPayment", ctxutil.GetStoredFields(ctx)...)
	return
}
