package provider

import (
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/trust"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/models"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"

	"github.com/blang/semver/v4"

	"context"
	"fmt"
	"sort"
)

// https://st.yandex-team.ru/GMSP-388#6001b0372d39482e7a29cf31
func setLastPaidCardsToTop(paymentMethods models.BoundPaymentMethods) models.BoundPaymentMethods {
	result := make(models.BoundPaymentMethods, 0)
	lastServicePaidIndex := -1
	lastPaidIndex := -1
	for i, paymentMethod := range paymentMethods {
		if paymentMethod.LastServicePaid > 0 {
			lastServicePaidIndex = i
		} else if paymentMethod.LastPaid > 0 {
			lastPaidIndex = i
		}
	}

	if lastServicePaidIndex >= 0 {
		result = append(result, paymentMethods[lastServicePaidIndex])
	}

	if lastPaidIndex >= 0 {
		result = append(result, paymentMethods[lastPaidIndex])
	}

	for i, paymentMethod := range paymentMethods {
		if i != lastServicePaidIndex && i != lastPaidIndex {
			result = append(result, paymentMethod)
		}
	}

	return result
}

func (s *PaymentService) PaymentMethodsDefault(ctx context.Context, uid uint64, serviceToken string, paymentSystemsOptions models.PaymentSystemsOptions) (methods models.UserPaymentMethods, err error) {
	return s.PaymentMethods(ctx, uid, serviceToken, paymentSystemsOptions, models.ProductionOrder)
}

func (s *PaymentService) PaymentMethods(ctx context.Context, uid uint64, serviceToken string, paymentSystemsOptions models.PaymentSystemsOptions, env models.OrderEnvironment) (methods models.UserPaymentMethods, err error) {
	if len(serviceToken) == 0 {
		serviceToken = s.config.GetServiceToken(env, s.config.DefaultAcquirer)
	}
	ctx = ctxutil.WithServiceToken(ctx, serviceToken)
	logger := ctxutil.GetLogger(ctx)

	trustClient := s.getTrustClient(env)
	r, err := trustClient.GetPaymentMethods(ctx, serviceToken, uid)
	if err != nil {
		return
	}

	logger.Info(
		fmt.Sprintf("Trust payment methods: %v", r),
		ctxutil.GetStoredFields(ctx)...,
	)

	boundPaymentMethodsWhitelist := make(map[string]struct{})
	for _, s := range s.config.BoundPaymentMethodsWhitelist {
		boundPaymentMethodsWhitelist[s] = struct{}{}
	}

	methods.PaymentMethods = make(models.BoundPaymentMethods, 0)
	for _, m := range r.BoundPaymentMethods {
		if _, ok := boundPaymentMethodsWhitelist[m.PaymentMethod]; ok {
			m.VerifyCvv = true
			methods.PaymentMethods = append(methods.PaymentMethods, convertTrustBoundPaymentMethod(s.config, &m))
		}
	}

	sort.Sort(sort.Reverse(methods.PaymentMethods))
	methods.PaymentMethods = setLastPaidCardsToTop(methods.PaymentMethods)
	methods.PaymentMethods = filterBoundPaymentMethods(ctx, methods.PaymentMethods)

	for _, em := range r.EnabledPaymentMethods {
		methods.EnabledPaymentMethods = append(methods.EnabledPaymentMethods, models.EnabledPaymentMethods(em))
	}
	gpayEnabled := isGPayEnabled(s.config, uid, serviceToken)
	logger.Info(fmt.Sprintf("GPay enabled=%t for uid=%v", gpayEnabled, uid))
	applePayEnabled := isApplePayEnabled(s.config, uid, serviceToken)
	logger.Info(fmt.Sprintf("ApplePay enabled=%t for uid=%v", applePayEnabled, uid))
	methods.EnabledPaymentMethods = filterEnabledPaymentMethods(methods.EnabledPaymentMethods, s.config)
	methods.EnabledPaymentMethods = filterEnabledPaymentSystems(methods.EnabledPaymentMethods, paymentSystemsOptions, gpayEnabled, applePayEnabled)

	methods.GooglePaySupported = gpayEnabled && methods.PaymentSystemEnabled(models.GooglePay)
	methods.ApplePaySupported = applePayEnabled && methods.PaymentSystemEnabled(models.ApplePay)

	bankNames := s.resourceProvider.GetBankNames()
	mappedBanks := make(map[string]string)
	for i := range methods.PaymentMethods {
		method := &methods.PaymentMethods[i]
		originalName := method.CardBank
		method.MapBankName(bankNames)
		mappedBanks[originalName] = method.CardBank
	}

	logger.Info(
		fmt.Sprintf("Resolve bank names: %v", mappedBanks),
		ctxutil.GetStoredFields(ctx)...,
	)

	logger.Info(
		fmt.Sprintf("Resolved payment methods: %v", methods),
		ctxutil.GetStoredFields(ctx)...,
	)

	return
}

var (
	yabankEnabledVersion = semver.Version{
		Major: 3,
		Minor: 9,
		Patch: 0,
	}
)

func filterBoundPaymentMethods(ctx context.Context, boundPaymentMethods models.BoundPaymentMethods) models.BoundPaymentMethods {
	sdkSemverValue, _ := ctxutil.GetSdkSemverValue(ctx)

	result := make([]models.BoundPaymentMethod, 0, len(boundPaymentMethods))
	for _, boundPaymentMethod := range boundPaymentMethods {
		if boundPaymentMethod.PartnerInfo != nil && boundPaymentMethod.PartnerInfo.IsYabankCard {
			if sdkSemverValue == nil || sdkSemverValue.LT(yabankEnabledVersion) {
				continue
			}
		}

		result = append(result, boundPaymentMethod)
	}
	return result
}

func filterEnabledPaymentMethods(enabledPaymentMethods []models.EnabledPaymentMethods, config *Config) []models.EnabledPaymentMethods {
	result := make([]models.EnabledPaymentMethods, 0, len(enabledPaymentMethods))
	for _, enabledPaymentMethod := range enabledPaymentMethods {
		if config.HideSBPQR && enabledPaymentMethod.PaymentMethod == models.SBPQR.String() {
			continue
		}
		result = append(result, enabledPaymentMethod)
	}
	return result
}

func filterEnabledPaymentSystems(enabledPaymentMethods []models.EnabledPaymentMethods, paymentSystemsOptions models.PaymentSystemsOptions, gpayEnabled bool, applePayEnabled bool) []models.EnabledPaymentMethods {
	result := make([]models.EnabledPaymentMethods, 0, len(enabledPaymentMethods))
	for _, enabledPaymentMethod := range enabledPaymentMethods {
		paymentSystems := make([]string, 0, len(enabledPaymentMethod.PaymentSystems))
		for _, paymentSystem := range enabledPaymentMethod.PaymentSystems {
			if paymentSystem == models.ApplePay.String() && (!paymentSystemsOptions.ApplePayEnabled || !applePayEnabled) {
				continue
			}
			if paymentSystem == models.GooglePay.String() && (!paymentSystemsOptions.GooglePayEnabled || !gpayEnabled) {
				continue
			}
			paymentSystems = append(paymentSystems, paymentSystem)
		}

		enabledPaymentMethod.PaymentSystems = paymentSystems
		result = append(result, enabledPaymentMethod)
	}
	return result
}

func isGPayEnabled(config *Config, uid uint64, serviceToken string) bool {
	if uid == 0 {
		return false
	}
	enabled, ok := config.GooglePay.PerUID[uid]
	if ok {
		return enabled
	}

	enabled, ok = config.GooglePay.PerService[serviceToken]
	if ok {
		return enabled
	}

	return config.GooglePay.Enabled
}

func isApplePayEnabled(config *Config, uid uint64, serviceToken string) bool {
	if uid == 0 {
		return false
	}
	enabled, ok := config.ApplePay.PerUID[uid]
	if ok {
		return enabled
	}

	enabled, ok = config.ApplePay.PerService[serviceToken]
	if ok {
		return enabled
	}

	return config.ApplePay.Enabled
}

func isFamilyPayUnlimited(config *Config, limit int) bool {
	return uint32(limit) >= config.FamilyPay.UnlimitedBalanceInFractionalUnits
}

func convertTrustBoundPaymentMethod(config *Config, method *trust.BoundPaymentMethod) models.BoundPaymentMethod {
	var payerInfo *models.PayerInfo = nil
	if method.PayerInfo != nil {
		familyInfo := method.PayerInfo.FamilyInfo
		payerInfo = &models.PayerInfo{
			UID: method.PayerInfo.UID,
			FamilyInfo: models.FamilyInfo{
				FamilyID:  familyInfo.FamilyID,
				Expenses:  familyInfo.Expenses,
				Limit:     familyInfo.Limit,
				Frame:     familyInfo.Frame,
				Currency:  familyInfo.Currency,
				Unlimited: isFamilyPayUnlimited(config, familyInfo.Limit),
			},
		}
	}

	var partnerInfo *models.PartnerInfo = nil
	if method.PartnerInfo != nil {
		partnerInfo = &models.PartnerInfo{
			IsYabankCard:      method.PartnerInfo.IsYabankCard,
			IsYabankCardOwner: method.PartnerInfo.IsYabankCardOwner,
			IsFakeYabankCard:  method.PartnerInfo.IsFakeYabankCard,
		}
	}

	return models.BoundPaymentMethod{
		RegionID:                    method.RegionID,
		PaymentMethod:               method.PaymentMethod,
		BindingTS:                   method.BindingTS,
		RecommendedVerificationType: method.RecommendedVerificationType,
		Aliases:                     method.Aliases,
		Expired:                     method.Expired,
		CardBank:                    method.CardBank,
		System:                      method.System,
		ID:                          method.ID,
		CardCountry:                 method.CardCountry,
		PaymentSystem:               method.PaymentSystem,
		CardLevel:                   method.CardLevel,
		Holder:                      method.Holder,
		BindingSystems:              method.BindingSystems,
		Account:                     method.Account,
		CardID:                      method.CardID,
		VerifyCvv:                   method.VerifyCvv,
		IsSpasibo:                   method.IsSpasibo,
		LastPaid:                    method.LastPaid,
		LastServicePaid:             method.LastServicePaid,
		PayerInfo:                   payerInfo,
		PartnerInfo:                 partnerInfo,
	}
}
